#!/usr/bin/perl -w
#
# Copyright (c) 2007-2020 VMware, Inc.  All rights reserved.
#
# vmdeploytocluster.pl - deploy a VM from the svdc-template
#
# Adapted from /usr/lib/vmware-vcli/apps/vmdeploytocluster.pl for SPECvirt Datacenter 2021
#
# $1 = targetVM
# $2 = MAC address
# $3 = numvCPUS

use strict;
#use warnings;
#use 5.010;


use VMware::VILib;
use VMware::VIRuntime;
use FindBin;
use lib "$FindBin::Bin/../../";
use lib "/usr/lib/vmware-vcli/apps";

use XML::LibXML;
use AppUtil::VMUtil;
use AppUtil::XMLInputUtil;
#use AppUtil::HostUtil;
use perl::HostUtil;

$Util::script_version = "2.0";

sub check_missing_value;

my %opts = (
   cluster => {
      type => "=s",
      help => "The cluster name",
      required => 0,
   },
   templatename => {
      type => "=s",
      help => "The name of the source template",
      required => 0,
      default => "svdc-template",
   },
   targetVM => {
      type => "=s",
      help => "The name of the target VM",
      required => 1,
   },
   mac => {
      type => "=s",
      help => "The MAC address of the target VM",
      required => 1,
   },
   filename => {
      type => "=s",
      help => "The name of the configuration specification file",
      required => 0,
      default => "../sampledata/vmclone.xml",
   },
   customize_guest => {
      type => "=s",
      help => "Flag to specify whether or not to customize guest: yes,no",
      required => 0,
      default => 'no',
      #default => 'yes',
   },
   customize_vm => {
      type => "=s",
      help => "Flag to specify whether or not to customize VM: yes,no",
      required => 0,
      default => 'no',
      #default => 'yes',
   },
   schema => {
      type => "=s",
      help => "The name of the schema file",
      required => 0,
      default => "../schema/vmclone.xsd",
   },
   datastore => {
      type => "=s",
      help => "Name of the target datastore",
      required => 1,
   },
);

Opts::add_options(%opts);
Opts::parse();
Opts::validate(\&validate);
Util::connect();

$Data::Dumper::Sortkeys = 1; #Sort the keys in the output
$Data::Dumper::Deepcopy = 1; #Enable deep copies of structures
$Data::Dumper::Indent = 2; #Output in a reasonable style (but no array indexes)

clone_vm();

Util::disconnect();

sub clone_vm {

my $i=0;
my $host="";
my $cluster = Opts::get_option('cluster');

my $cluster_view = Vim::find_entity_view(view_type => 'ClusterComputeResource',
                                          filter => { name => $cluster });
unless($cluster_view) {
  die "Unable to locate cluster name: \"$cluster\"!";
}

my $clusterhost_views = Vim::get_views(mo_ref_array => $cluster_view->host, properties => ['name', 'runtime' ]);

#Get a host on the cluster that's not in maintenance mode to get datastore info

my $clusterHostsOnline = scalar (@$clusterhost_views);

foreach  (@$clusterhost_views) {
    $i++;
    ###print "Debug: Host.name[".$i."]=" . $_->name . "\n";
    $host = $_->name;
    if ($_->runtime->connectionState->val eq "connected") {
      if ( $_->runtime->inMaintenanceMode) {
        $clusterHostsOnline--;
      } else {
        last;
      }
    }
}

unless($clusterHostsOnline >= 1) {
  die "Hosts in cluster: \"$cluster\" are offline!";
}

my $host_view = Vim::find_entity_view(view_type => 'HostSystem',
                                         filter => {'name' => $host});
if (!$host_view) {
  Util::trace(0, "Host '$host' not found\n");
  return;
}

###
# Get source template and host views
###
   my $templatename_name = Opts::get_option('templatename');
#Util::trace (0, "\nSource template is '$templatename_name'.\n");
   my $templatename_views = Vim::find_entity_views(view_type => 'VirtualMachine',
                                        filter => {'name' =>$templatename_name});
   if(@$templatename_views) {
      foreach (@$templatename_views) {

            my $ds_name = Opts::get_option('datastore');
            my %ds_info = HostUtils::get_datastore(host_view => $host_view,
                                     datastore => $ds_name,
                                     disksize => get_disksize());

            if ($ds_info{mor} eq 0) {
               if ($ds_info{name} eq 'datastore_error') {
                  Util::trace(0, "\nDatastore $ds_name not available.\n");
                  return;
               }
               if ($ds_info{name} eq 'disksize_error') {
                  Util::trace(0, "\nThe free space available is less than the"
                               . " specified disksize or the host"
                               . " is not accessible.\n");
                  return;
               }
         }

###
# Creating relocation spec if moving targetVM
###
    my $relocate_spec = VirtualMachineRelocateSpec->new(datastore => $ds_info{mor},
                                          pool => $cluster_view->resourcePool);
    my $clone_name = Opts::get_option('targetVM');
    my $clone_spec ;
    my $config_spec;
    my $customization_spec;


    if ((Opts::get_option('customize_vm') eq "yes")
                && (Opts::get_option('customize_guest') ne "yes")) {
#Util::trace (0, "\n Clone OPTION 1.\n");
               $config_spec = get_config_spec();
               $clone_spec = VirtualMachineCloneSpec->new(powerOn => 0,template => 0,
                                                       location => $relocate_spec,
                                                       config => $config_spec,
                                                       );
    }elsif ((Opts::get_option('customize_guest') eq "yes")
                && (Opts::get_option('customize_vm') ne "yes")) {
#Util::trace (0, "\n Clone OPTION 2.\n");
               $customization_spec = VMUtils::get_customization_spec
                                              (Opts::get_option('filename'));
               $clone_spec = VirtualMachineCloneSpec->new(
                                                   powerOn => 0,
                                                   template => 0,
                                                   location => $relocate_spec,
                                                   customization => $customization_spec,
                                                   );
    }elsif ((Opts::get_option('customize_guest') eq "yes")
                && (Opts::get_option('customize_vm') eq "yes")) {
#Util::trace (0, "\n Clone OPTION 3.\n");

               $customization_spec = VMUtils::get_customization_spec
                                              (Opts::get_option('filename'));
               $config_spec = get_config_spec();
               $clone_spec = VirtualMachineCloneSpec->new(
                                                   powerOn => 0,
                                                   template => 0,
                                                   location => $relocate_spec,
                                                   customization => $customization_spec,
                                                   config => $config_spec,
                                                   );
    }else {
#Util::trace (0, "\n Clone OPTION 4.\n");

               $clone_spec = VirtualMachineCloneSpec->new(
                                                   powerOn => 0,
                                                   template => 0,
                                                   location => $relocate_spec,
                                                   );
    }
Util::trace (0, "\nCloning '$templatename_name' to '$clone_name' ...\n");

    eval {
          $_->CloneVM(folder => $_->parent,
                         name => Opts::get_option('targetVM'),
                         spec => $clone_spec);
               Util::trace (0, "\nClone '$clone_name' from '$templatename_name' created successfully.\n");
        };

        if ($@) {
               if (ref($@) eq 'SoapFault') {
                  if (ref($@->detail) eq 'FileFault') {
                     Util::trace(0, "\nFailed to access the VM files\n");
        }elsif (ref($@->detail) eq 'InvalidState') {
                     Util::trace(0,"The operation is not allowed "
                                   ."in the current state.\n");
        }elsif (ref($@->detail) eq 'NotSupported') {
                     Util::trace(0," Operation is not supported by the "
                                   ."current agent \n");
        }elsif (ref($@->detail) eq 'VmConfigFault') {
                     Util::trace(0,
                     "VM is not compatible with the destination host.\n");
        }elsif (ref($@->detail) eq 'InvalidPowerState') {
                     Util::trace(0,
                     "The attempted operation cannot be performed "
                     ."in the current state.\n");
        }elsif (ref($@->detail) eq 'DuplicateName') {
                     Util::trace(0,
                     "The VM '$clone_name' already exists\n");
        }elsif (ref($@->detail) eq 'NoDisksToCustomize') {
                     Util::trace(0, "\nThe VM has no virtual disks that"
                                  . " are suitable for customization or no guest"
                                  . " is present on given VM" . "\n");
        }elsif (ref($@->detail) eq 'HostNotConnected') {
                     Util::trace(0, "\nUnable to communicate with the remote host, "
                                    ."since it is disconnected" . "\n");
        }elsif (ref($@->detail) eq 'UncustomizableGuest') {
                     Util::trace(0, "\nCustomization is not supported "
                                    ."for the guest operating system" . "\n");
        }else {
                Util::trace (0, "Fault" . $@ . ""   );
                }
              }
         else {
                  Util::trace (0, "Fault" . $@ . ""   );
               }
           }
      }
   }
   else {
        Util::trace (0, "\nNo template found with name '$templatename_name'\n");
 }

### 
# Set MAC address on newly cloned VM
###

set_mac();

}  #### end clone_vm

sub set_mac() 
{

###
# Get targetVM view and MAC address
###

   my $mac = Opts::get_option('mac');
#   Util::trace (0, "\nMAC address is '$mac'. Assigning to target VM ... \n");

   my $targetVM_name = Opts::get_option('targetVM');
   my $targetVM_view = Vim::find_entity_view(view_type => 'VirtualMachine',
                                        filter => {'name' =>$targetVM_name});
   if (!$targetVM_view) 
   {
      Util::trace(0, "Target VM '$targetVM_name' not found\n");
      return;
   }

   my $devices = $targetVM_view->config->hardware->device;

   my ($key,$unitNumber,$backing,$controllerKey,$type);

   $type = "null";

   foreach my $device (@$devices)
   {
      if($device->isa("VirtualEthernetCard")) 
      {
         if ($device->isa('VirtualE1000')) {
             $type = "VirtualE1000";
         }
         elsif($device->isa('VirtualPCNet32')) {
             $type = "VirtualPCNet32";
         }
         elsif($device->isa('VirtualVmxnet3')) {
#             Util::trace (0, "\nFound vNIC: VirtualVmxnet3'\n");
             $type = "VirtualVmxnet3";
         }
         elsif($device->isa('VirtualVmxnet2')) {
             $type = "VirtualVmxnet2";
         }
         $key = $device->key;
         $controllerKey = $device->controllerKey;
         $unitNumber = $device->unitNumber;
         $backing = $device->backing;

         my $specOp = VirtualDeviceConfigSpecOperation->new('edit');
         my $virtualdevice;

         if($type eq "VirtualVmxnet3") {
            $virtualdevice = VirtualVmxnet3->new(
                controllerKey => $controllerKey,
                key => $key,
                backing => $backing,
                unitNumber => $unitNumber,
                macAddress => $mac,
                addressType => 'Manual'
                 );
         }
        my $virtdevconfspec = VirtualDeviceConfigSpec->new(
	   device => $virtualdevice,
	   operation => $specOp
   	);

   	my $virtmachconfspec = VirtualMachineConfigSpec->new(
      	   deviceChange => [$virtdevconfspec],
   	);

   	eval {
      	   $targetVM_view->ReconfigVM_Task( spec => $virtmachconfspec );
      	   Util::trace(0,"\nVirtual machine '" . $targetVM_view->name
              . "' MAC address reconfigured successfully.\n");
   	};

   	if ($@) 
	{
      	   Util::trace(0, "\nReconfiguration failed: ");
           if (ref($@) eq 'SoapFault') 
	   {
              if (ref($@->detail) eq 'TooManyDevices') {
                 Util::trace(0, "\nNumber of virtual devices exceeds "
                     . "the maximum for a given controller.\n");
              }
	      elsif (ref($@->detail) eq 'InvalidDeviceSpec') {
            	Util::trace(0, "The Device configuration is not valid\n");
            	Util::trace(0, "\nFollowing is the detailed error: \n\n$@");
              }
	      elsif (ref($@->detail) eq 'FileAlreadyExists') 
	      {
                 Util::trace(0, "\nOperation failed because file already exists");
              }
	      else 
	      {
                 Util::trace(0, "\n" . $@ . "\n");
              }
           }
        }
   	else 
	{
           Util::trace(0, "\n" . $@ . "\n");
   	}
     }
  }
#   Util::trace (0, "\nReassigned MAC address on '$targetVM_name'\n");
}


###
# Gets the config_spec if customizing the memory, number of cpus
#   and returns the spec
###
sub get_config_spec() {

   my $parser = XML::LibXML->new();
   my $tree = $parser->parse_file(Opts::get_option('filename'));
   my $root = $tree->getDocumentElement;
   my @cspec = $root->findnodes('Virtual-Machine-Spec');
   my $templatename ;
   my $targetVM ;
   my $guestid;
   my $datastore;
   my $disksize = 4096;  # in KB;
   my $memory = 256;  # in MB;
   my $num_cpus = 1;
   my $nic_network;
   my $nic_poweron = 1;

   foreach (@cspec) {
   
      if ($_->findvalue('Guest-Id')) {
         $guestid = $_->findvalue('Guest-Id');
      }
      if ($_->findvalue('Memory')) {
         $memory = $_->findvalue('Memory');
      }
      if ($_->findvalue('Number-of-CPUS')) {
         $num_cpus = $_->findvalue('Number-of-CPUS');
#	 Util::trace (0, "\n Number-of-CPUS = '$num_cpus'\n");
      }
      $targetVM = Opts::get_option('targetVM');
   }

   my $vm_config_spec = VirtualMachineConfigSpec->new(
                                                  name => $targetVM,
                                                  memoryMB => $memory,
                                                  numCoresPerSocket => $num_cpus,
                                                  numCPUs => $num_cpus,
                                                  guestId => $guestid );

   return $vm_config_spec;
 }

sub get_disksize {
   my $disksize = 4194304;
   my $parser = XML::LibXML->new();
   
   eval {
      my $tree = $parser->parse_file(Opts::get_option('filename'));
      my $root = $tree->getDocumentElement;
      my @cspec = $root->findnodes('Virtual-Machine-Spec');

      foreach (@cspec) {
         $disksize = $_->findvalue('Disksize');
      }
   };
   return $disksize;
}

# check missing values of mandatory fields
sub check_missing_value {
   my $valid= 1;
   my $filename = Opts::get_option('filename');
   my $parser = XML::LibXML->new();
   my $tree = $parser->parse_file($filename);
   my $root = $tree->getDocumentElement;
   my @cust_spec = $root->findnodes('Customization-Spec');
   my $total = @cust_spec;
   if (!$cust_spec[0]->findvalue('Virtual-Machine-Name')) {
      Util::trace(0,"\nERROR in '$filename':\n computername value missing ");
      $valid = 0;
   }
   return $valid;
}

sub validate {
   my $valid= 1;
   if ((Opts::get_option('customize_vm') eq "yes")
                || (Opts::get_option('customize_guest') eq "yes")) {

      $valid = XMLValidation::validate_format(Opts::get_option('filename'));
      if ($valid == 1) {
         $valid = XMLValidation::validate_schema(Opts::get_option('filename'),
                                             Opts::get_option('schema'));
         if ($valid == 1) {
            $valid = check_missing_value();
         }
      }
   }

   if (Opts::option_is_set('customize_vm')) {
       if ((Opts::get_option('customize_vm') ne "yes")
             && (Opts::get_option('customize_vm') ne "no")) {
          Util::trace(0,"\nMust specify 'yes' or 'no' for customize_vm option");
          $valid = 0;
       }
       
   }
   if (Opts::option_is_set('customize_guest')) {
       if ((Opts::get_option('customize_guest') ne "yes")
             && (Opts::get_option('customize_guest') ne "no")) {
          Util::trace(0,"\nMust specify 'yes' or 'no' for customize_guest option");
          $valid = 0;
       }
   }
   return $valid;
}
__END__

=head1 NAME

vmdeploytocluster.pl - Perform clone operation on VM and
             customize operation on both VM and the guest.

=head1 SYNOPSIS

 vmdeploytocluster.pl [options]

=head1 DESCRIPTION

VI Perl command-line utility allows you to clone a VM. You 
can customize the VM or the guest operating system as part 
of the clone operation.

=head1 OPTIONS

=head2 GENERAL OPTIONS

=over

=item B<cluster>

Required. Name of the destination cluster.

=item B<templatename>

Required. Name of the VM whose clone is to be created.

=item B<targetVM>

Required. Name of the clone VM which will be created.

=item B<datastore>

Required. Name of a datastore. 

=back

=head2 CUSTOMIZE GUEST OPTIONS

=over

=item B<customize_guest>

Required. Customize guest is used to customize the network settings of the guest
operating system. Options are Yes/No.

=item B<filename>

Required. It is the name of the file in which values of parameters to be
customized is written e.g. --filename  clone_vm.xml.

=item B<schema>

Required. It is the name of the schema which validates the filename.

=back

=head2 CUSTOMIZE VM OPTIONS

=over

=item B<customize_vm>

Required. customize_vm is used to customize the VM settings
like disksize, memory. If yes is written it will be customized.

=item B<filename>

Required. It is the name of the file in which values of parameters to be
customized is written e.g. --filename  clone_vm.xml.

=item B<schema>

Required. It is the name of the schema which validates the filename.

=back

=head2 INPUT PARAMETERS

=head3 GUEST CUSTOMIZATION

The parameters for customizing the guest os are specified in an XML
file. The structure of the input XML file is:

 <Specification>
  <Customization-Spec>
  </Customization-Spec>
 </Specification>

Following are the input parameters:

=over

=item B<Auto-Logon>

Required. Flag to specify whether auto logon should be enabled or disabled.

=item B<Virtual-Machine-Name>

Required. Name of the VM to be created.

=item B<Timezone>

Required. Time zone property of guest OS.

=item B<Domain>

Required. The domain that the VM should join.

=item B<Domain-User-Name>

Required. The domain user account used for authentication.

=item B<Domain-User-Password>

Required. The password for the domain user account used for authentication.

=item B<Full-Name>

Required. User's full name.

=item B<Orgnization-Name>

Required. User's organization.

=back

=head3 VIRTUAL MACHINE CUSTOMIZATION

The parameters for customizing the VM are specified in an XML
file. The structure of the input XML file is:

   <Specification>
    <Config-Spec-Spec>
       <!--Several parameters like Guest-Id, Memory, Disksize, Number-of-CPUS etc-->
    </Config-Spec>
   </Specification>

Following are the input parameters:

=over

=item B<Guest-Id>

Required. Short guest operating system identifier.

=item B<Memory>

Required. Size of a VM's memory, in MB.

=item B<Number-of-CPUS>

Required. Number of virtual processors in a VM.

=back

See the B<vmcreate.pl> page for an example of a VM XML file.

=head1 EXAMPLES

Making a clone without any customization:

 perl vmdeploytocluster.pl --username username --password mypassword
                 --templatename DVM1 --targetVM DVM99
                 --cluster CL1 --url https://<ipaddress>:<port>/sdk/webService

If datastore is given:

 perl vmdeploytocluster.pl --username username --password mypassword
                 --templatename DVM1 --targetVM DVM99
                 --cluster CL1 --url https://<ipaddress>:<port>/sdk/webService --datastore storage1

Making a clone and customizing the VM:

 perl vmdeploytocluster.pl --username myusername --password mypassword
                 --templatename DVM1 --targetVM Clone_VM
                 --cluster CL1 --url https://<ipaddress>:<port>/sdk/webService --customize_vm yes
                 --filename clone_vm.xml --schema clone_schema.xsd

Making a clone and customizing the guestOS:

 perl vmdeploytocluster.pl --username myuser --password mypassword --operation clone
                 --templatename DVM1 --targetVM DVM99
                 --cluster CL1 --url https://<ipaddress>:<port>/sdk/webService --customize_guest yes
                 --filename clone_vm.xml --schema clone_schema.xsd

Making a clone and customizing both guestos and VM:

 perl vmdeploytocluster.pl --username myuser --password mypassword
                 --templatename DVM1 --targetVM DVM99
                 --cluster CL1 --url https://<ipaddress>:<port>/sdk/webService --customize_guest yes
                 --customize_vm yes --filename clone_vm.xml --schema clone_schema.xsd

All the parameters which are to be customized are written in the vmclone.xml file.

=head1 SUPPORTED PLATFORMS

All operations supported on VirtualCenter 2.0.1 or later.

To perform the clone operation, you must connect to a VirtualCenter server.



