#!/usr/bin/env python
########################################################################
# Copyright 2016 VMware, Inc.  All rights reserved.
# -- VMware Confidential
########################################################################

"""Run any scripts generated by the kickstart mechanism

   The firstboot data is captured by the installer. Any scripts that need
   to be executed via the kickstart mechanism are added to a special
   directory.

   If UEFI secure boot is enabled, none of these scripts are executed.

"""

import os
import subprocess
import sys
from logging import (
   CRITICAL,
   DEBUG,
   FileHandler,
   Formatter,
   getLogger,
   Handler,
   INFO,
   StreamHandler,
)


secureBootCmd = ['/usr/lib/vmware/secureboot/bin/secureBoot.py', '-s']
scriptsDir = '/var/lib/vmware/firstboot'
loggerName = 'kickstartScript'
logFile = '/var/log/kickstart.log'

def setupLogger():
   """Create and initialize a logger, ready for use.

   """
   logger = getLogger(loggerName)
   logger.setLevel(INFO)

   formatter = Formatter('%(asctime)s %(levelname)s %(message)s')

   streamHandler = StreamHandler()
   streamHandler.setFormatter(formatter)
   streamHandler.setLevel(INFO)
   logger.addHandler(streamHandler)

   try:
      fileHandler = FileHandler(logFile)
      fileHandler.setFormatter(formatter)
      fileHandler.setLevel(INFO)
      logger.addHandler(fileHandler)
   except Exception as e:
      logger.warning("Failed to set up log file %s: \"%s\"", logFile, e)

   return logger



def main(argv=None):
   secureBootEnabled = True

   logger = setupLogger()
   try:
      output, error = subprocess.Popen(secureBootCmd, stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE).communicate()
      if output.decode().startswith("Disabled"):
         secureBootEnabled = False
   except Exception as e:
      logger.error("Error in getting Secure Boot status: %s" % e)
      # ignore errors and fall back to secureBootEnabled = True
      pass

   for root, dirs, files in os.walk(scriptsDir):
      for f in files:
         path = os.path.join(root, f)
         if os.access(path, os.X_OK):
            if secureBootEnabled:
               logger.info("UEFI Secure Boot Enabled, skipping execution of %s" % path)
            else:
               logger.info("Running %s" % path)
               try:
                  output, error = subprocess.Popen(path, stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE).communicate()
               except Exception as e:
                  logger.error("Failed to run %s: %s: %s" % (path, error, e))
                  pass

if __name__ == '__main__':
   sys.exit(main())
