#!/bin/sh

REPORTERVERSION="09282022"

if [ ! -e /bin/vmware ] ; then
   printf "\n$0 must be run on an ESXi host.  To run the Reporter," 
   printf "\nplease execute Reporter.sh on svdc-director.\n"
   exit 1
fi

# Copy host manifest from temporary directory to vm-support manifest directory
if [ -e /tmp/Reporter-Host.mfx ] ; then
   cp /tmp/Reporter-Host.mfx /etc/vmware/vm-support
else
   printf "\nThe Host manifest does not exist.  To run the Reporter," 
   printf "\nplease execute Reporter.sh on svdc-director.\n"
   exit 1
fi

rm -f /tmp/ReporterFile.txt 2>/dev/null  # clean up old temporary files from unclean exit

RAND=$(date +"%N" | cut -c 1-7)
DATE=$DATE$RAND
if [[ "$RAND" != "%N" ]] ; then  
   DATE=$(date +"%Y%m%d-%H%M")$RAND
fi
#
# Choose LUN for reporter directory that has enough disk space for the reporter file.
# ex. REPORTERDIR=/vmfs/volumes/LUN1/reporter-20220719-22184564225
# 
REPORTERDIR=/vmfs/volumes/$(ls /vmfs/volumes 2>/dev/null |grep -v " "|sort -u |tail -1)/reporter-$DATE # backup lun in case all other luns fail requirements
LUNS=$(df -m | awk '{print $4 $6 $7}' | sed '1d') # Lists space free on volume in MB.
TESTFILE=testfile$RAND

for i in $LUNS; do
   LUN=$(echo $i | sed 's/^.*\/vmfs\/volumes\///')
   MBFREE=$(echo ${i%%\/*})
   # Test there are no spaces in LUN name
   if [ "$(ls /vmfs/volumes 2>/dev/null | grep "$LUN" | grep -v " ")" != "" -a "$LUN" != "" ] ; then
      cd /vmfs/volumes/$LUN 2>/dev/null
      # Test writeability of LUN
      touch $TESTFILE 2>/dev/null
      if [ $? -eq 0 ] ; then # if touch completed successfully
         if [ -z $REPORTERDIR ] ; then # backup reporter directory has not been chosen
            # This lun passes our two requirements - writable and no spaces. If we are unable to find a lun that passes space free, fall back on this one.
            REPORTERDIR=/vmfs/volumes/$LUN/reporter-$DATE 
         fi
         if [ -e $TESTFILE ] ; then
            rm -f $TESTFILE 2>/dev/null
            cd - >/dev/null            
            # Test adequate free space on LUN
            if [ "$MBFREE" -gt 30000 ] ; then # Ensure that LUN has at least 30,000 MB (29.3GB) of space free for reporter file(s).
               REPORTERDIR=/vmfs/volumes/$LUN/reporter-$DATE          
               # Test whether another reporter directory exists on this lun
               if [ "$LUN" != "" ] && [ -z "$(find /vmfs/volumes/$LUN/ -name "reporter*" -type d)" ] ; then
                  REPORTERDIR=/vmfs/volumes/$LUN/reporter-$DATE
                  break
               fi
            fi
         fi
      fi
	fi
done

# Hostname will be part of reporter package filename.
HOSTNAME=$(hostname -s 2>/dev/null)
# If hostname is localhost, use hostname if passed from director
if [ "$HOSTNAME" = "localhost" ] && [ -n "$1" ] ; then
   HOSTNAME=$1
fi

# Append two random numbers to working directory name in case of hostname conflict. 
RAND=$(date +"%N" | cut -c 1-2)

if [[ "$RAND" != "%N" ]] ; then	
	DATE=$(date +"%Y%m%d-%H%M")$RAND
	# If there is a conflict in hostnames and random numbers, add more random numbers to filename 
	if [ -d $REPORTERDIR/$HOSTNAME-$DATE ] ; then
			RAND=$(date +"%N" | cut -c 1-7)
			DATE=$DATE$RAND
	fi
fi
SUBMITDIR=$HOSTNAME-$DATE         # ex. suthost4-20220719-221868
WORKDIR=$REPORTERDIR/$SUBMITDIR   # ex. /vmfs/volumes/LUN1/reporter-20220719-22184564225/suthost4-20220719-221868
VMSUPPORTTEMPDIR=$REPORTERDIR/vm-support-$SUBMITDIR  # ex. /vmfs/volumes/LUN1/reporter-20220719-22184564225/vm-support-suthost4-20220719-221868

/usr/lib/vmware/osfs/bin/osfs-mkdir $REPORTERDIR
mkdir -pm 755 $WORKDIR
mkdir -pm 755 $VMSUPPORTTEMPDIR

if [ ! -d "$WORKDIR" ] ; then
   printf "\nUnable to create working directory $WORKDIR. Please contact Support.\n"
   exit 1
fi

############################################################################
# Collect and extract vm-support file
############################################################################

# Exclude (-e) specific manifests specified below from vm-support execution for the host.
# These manifests are replaced by Reporter-Host.mfx which has already been transferred by Reporter.sh.
# This manifest is run by default so it does not have to be specified.
# Additionally, if vSAN is not present on the host, exclude all vSAN manifests.
VSAN=$(localcli storage filesystem list -i | grep -i vsan)
if [ "$VSAN" ] ; then
   OUTPUTSTR=$(vm-support -e "System:Base Logs:System VirtualMachines:logs System:CoreDumps Storage:VSANTraces" -w $VMSUPPORTTEMPDIR |tail -1)
else
   OUTPUTSTR=$(vm-support -e "System:Base Logs:System VirtualMachines:logs System:CoreDumps Storage:VSAN Storage:VSANFileAnalyticsDump Storage:VSANFileSvc Storage:VSANHealth Storage:VSANIscsiTarget Storage:VSANPerfStats Storage:VSANPerfSvc Storage:VSANTraces" -w $VMSUPPORTTEMPDIR |tail -1)
fi
rm -f /etc/vmware/vm-support/Reporter-Host.mfx
# Extract vm-support filename from vm-support output string
VMSUPPORTFILE=$(echo ${OUTPUTSTR#T*$VMSUPPORTTEMPDIR\/} | sed 's/.$//') # ex. esx-suthost4-2022-07-19--22.18-38128342.tgz
VMSUPPORTDIR=$(echo ${VMSUPPORTFILE%\.tgz}) # subdirectory created when VMSUPPORTFILE extracted, ex. esx-suthost4-2022-07-19--22.18-38128342

# Extract VMSUPPORTFILE to VMSUPPORTTEMPDIR
tar -xzf $VMSUPPORTTEMPDIR/$VMSUPPORTFILE -C $VMSUPPORTTEMPDIR
rm -f $VMSUPPORTTEMPDIR/$VMSUPPORTFILE

############################################################################
# Post-process extracted vm-support files/directories
############################################################################
cd $VMSUPPORTTEMPDIR/$VMSUPPORTDIR # cd into newly extracted directory

# Change all UUID in /vmfs/volumes directory names to Volume Name instead
# parse Volume names vs UUID
localcli storage vmfs extent list | sed '1,2d' | while read LINE ; do
   VOLNAME=$(echo $LINE | awk {'print $1'})
   # sed command: make sure expected UUID matches 8-8-4-12 character pattern. Error checks in case volume name has a space.
   UUID=$(echo $LINE | awk {'print $2'} | sed '/^[0-9,a-f]\{8\}-[0-9,a-f]\{8\}-[0-9,a-f]\{4\}-[0-9,a-f]\{12\}/!d')      
   # if UUID fits UUID format,
   if [ "$UUID" ] ; then
      ls vmfs/volumes 2> /dev/null | while read DIRNAME ; do
         if [ "$DIRNAME" = "$UUID" ] ; then
            if [[ "$VMSUPPORTDIR" && "$VMSUPPORTTEMPDIR" ]] ; then 	# confirm that these variables are not empty before issuing file commands
               mkdir $VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/$VOLNAME 2> /dev/null # if VOLNAME already exists, this will throw back an error, so ignore it.
               mv $VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/$DIRNAME/* $VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/$VOLNAME 2> /dev/null 
               rm -rf $VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/$DIRNAME
            else
               printf "Unable to rename LUN names.\n" 
            fi
         fi
      done
   fi
done

#
# Create list of all running VMs
#
VMID=$(vim-cmd vmsvc/getallvms 2>/dev/null | awk '$1 ~ /^[0-9]+$/ {print $1}')
echo Powered on VMs are > commands/all-powered-on-VMs.txt
for i in $VMID; do
STATE=$(vim-cmd vmsvc/power.getstate $i 2>/dev/null | tail -1 | awk '{print $2}')
   if [[ "$STATE" = "on" ]] ; then
         VMNAME=$(vim-cmd vmsvc/get.summary $i 2>/dev/null | grep name | awk '{print $3}' | sed 's/\"\|,//g')
      if [[ $VMNAME ]] ; then
         echo $VMNAME >> commands/all-powered-on-VMs.txt
      fi
   elif [[ "$STATE" = "off" ]] ; then
      # If VM just got migrated off this host, $STATE will not be on or off. We cannot retrieve its data, like VMPATH. Cannot delete VMDIR.
      # In this case, we may find directories for the migrated VM remaining in /vmfs/volumes/
      VMPATH=$(vim-cmd vmsvc/getallvms 2>/dev/null | grep "^$i\>" | sed -e 's/^.* \[.*] //' -e 's/.vmx.*$/.vmx/') # sed captures VM folder names with spaces in them
      VMDIR=$(dirname "$VMPATH" 2>/dev/null) 
      VMLUN=$(vim-cmd vmsvc/getallvms 2>/dev/null | grep "^$i\>" | sed -e 's/^.*\[//' -e 's/\].*//')
      # Handle vSAN, which is a special case
      if echo $VMLUN | grep "vsanDatastore" - > /dev/null; then
         VMLUN=$(readlink -v "/vmfs/volumes/$VMLUN") # replace vsanDatastore symlink with real name, i.e., vsan:527e8c8d63f5c574-b1e666b0482f5400 
      fi
      if [[ "$VMSUPPORTTEMPDIR" && "$VMSUPPORTDIR" && "$VMLUN" && "$VMDIR" ]] ; then 	# confirm that these variables are not empty before issuing rm command
      case $VMDIR in
            *) rm -r "$VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/$VMLUN/$VMDIR" 2>/dev/null ;; # otherwise, delete the powered off VM's directory
      esac
      else
         printf "Unable to remove powered off VM files from "$VMSUPPORTDIR/vmfs/volumes/$VMLUN/$VMDIR".\n"
      fi
   fi
done  
if [ "$VMNAME" = "" ] ; then
   echo No VMs are powered on. > commands/all-powered-on-VMs.txt
fi

# Remove empty directories
if [[ "$VMSUPPORTTEMPDIR" && "$VMSUPPORTDIR" ]] ; then
   find "$VMSUPPORTTEMPDIR/$VMSUPPORTDIR/vmfs/volumes/" -type d -exec rmdir {} + 2>/dev/null
else
   printf "Unable to remove empty directories.\n"
fi

# Rename directory and recompress as .tar.xz
cd $VMSUPPORTTEMPDIR
mv $VMSUPPORTDIR/* $REPORTERDIR/$SUBMITDIR # Change default vm-support name to friendly SUBMITDIR name
cd $REPORTERDIR
SUBMITFILE=$HOSTNAME-$DATE.tar.xz         # ex. suthost4-20220719-221868.tar.xz
tar -cJf $SUBMITFILE $SUBMITDIR	         # Compress SUBMITDIR subdirectory into SUBMITFILE in REPORTERDIR

# Cleanup
rm -rf $SUBMITDIR 2>/dev/null # Remove extracted vm-support copy
rm -rf $VMSUPPORTTEMPDIR 2>/dev/null

# Publish location of reporter package directory/filename for calling scripts & harness
# ex. /vmfs/volumes/LUN1/reporter-20220719-22184564225/suthost4-20220719-221868.tar.xz
printf "$REPORTERDIR/$SUBMITFILE\n" >> /tmp/ReporterFile.txt 2>/dev/null

