#!/bin/bash
#
#     Collects various configuration and log files for benchmark disclosure 
#     with VMware vSphere. This script, in combination with its 
#     companion scripts Reporter-Host.sh and Reporter-VC.sh,
#     prepares output from the VMware vSphere support scripts vm-support and
#     vc-support for benchmark submission.
#
##############################################################################
REPORTERVERSION="09282022"

# Set up environment variables
. /export/home/cp/bin/getVars.sh
SDKDIR="/export/home/cp/config/workloads/specvirt/HV_Operations/${virtVendor}"

HOSTNAME=$1
runfolder=`ls -l /export/home/cp/results/specvirt|grep ^d|awk '{print $9}'|egrep '^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{2}-[0-9]{2}-[0-9]{2}'|tail -1`
RESULTDIR=/export/home/cp/results/specvirt/$runfolder/support
OUTPUT_FAIL=0
STARTDIR=$(pwd)
SSHCOMMAND="ssh -o LogLevel=ERROR"
SCPCOMMAND="scp -o LogLevel=ERROR"

printUsage () {
   printf "\nUsage: bash $0 HOSTNAME [result_dir] [temp_dir]"
   printf "\nExample 1: bash $0 esx-sut-host1   /export/home/cp/results/Results_20220906162031 /datastore/tmp\n"
   printf "\nExample 2: bash $0 vcsa.vmware.com /export/home/cp/results/results/Results_20220906162031\n"
   printf "\nRuns Reporter script on an ESXi host or vCenter Server.\n"
   printf "\n   HOSTNAME : Target ESXi or vCenter Server hostname/IP address with passwordless SSH enabled\n"
   printf "\n   result_dir : Reporter results files <hostname>-<datestamp>.tar.xz and"
   printf "\n              ReporterResult-<hostname>.txt will be written to this"
   printf "\n              directory or the current directory by default.\n"
   printf "\n   temp_dir : Temporary files will be created in this directory, or"
   printf "\n              /datastore/tmp by default.\n"
   printf "\n   Reporter.sh is a bash script that must be run from a"
   printf "\n   terminal on the svdc-director.\n\n"
}


###############################################################################
# Error checking
###############################################################################
if [ ! $BASH_VERSION ] ; then
   printf "Please run this script with bash, i.e. bash $0\n"
   printUsage; exit 1
fi
if [ -z "$1" ]; then
   printf "Missing HOSTNAME.\nPlease specify HOSTNAME as a target ESXi or vCenter Server hostname/IP address.\n"
   printUsage; exit 1
fi
if [ "$1" = "localhost" ]; then
   printf "Please specify HOSTNAME as a target ESXi or vCenter Server hostname/IP address.\n"
   printUsage; exit 1
fi
#if [ -z "$2" ]; then
#   RESULTDIR=$PWD
#fi
if [ ! -e $SDKDIR/Reporter-Host.sh ] || [ ! -e $SDKDIR/Reporter-VC.sh ]; then
   printf "Please ensure Reporter-Host.sh and Reporter-VC.sh are located in $SDKDIR.\n"
   printUsage; exit 1
fi

# Find short hostname from host, set up path names 
HOSTNAME_SHORT=$($SSHCOMMAND $HOSTNAME "hostname -s")
# If short hostname is localhost, revert to original hostname
if [ "$HOSTNAME_SHORT" = "localhost" ] ; then
   HOSTNAME_SHORT=$1
fi
# Append random numbers to working directory name in case of hostname conflict. 
RAND=$(date +"%N" | cut -c 1-7)
DATE=$(date +"%Y%m%d-%H%M")$RAND
SUBMITFILE=$HOSTNAME_SHORT-$DATE.tar.xz
WORKDIR=/datastore/tmp/$HOSTNAME_SHORT-$DATE
mkdir -p $WORKDIR
# Set up temporary directory; sanitize temp directory argument $3
if [ ! -z "$3" ]; then
   WORKDIR=$3  
   # Replace any relative path (. at beginning of string) with the absolute path.
   WORKDIR=${WORKDIR/#\./$STARTDIR}
   # If there is a trailing '/', delete it
   WORKDIR=${WORKDIR/%\//}
   
   if [ ! -d $WORKDIR ] ; then 
      printf "Temporary output directory $WORKDIR does not exist.\n"
      printUsage; exit 1 
   else
      WORKDIR=$WORKDIR/$HOSTNAME_SHORT-$DATE
   fi
fi

OUTPUT_ERR="${WORKDIR}/output_err.txt"
OUTPUT_WARN="${WORKDIR}/output_warn.txt"

# Sanitize results directory argument
# Replace any relative path (. at beginning of string) with the absolute path.
RESULTDIR=${RESULTDIR/#\./$STARTDIR}
# If there is a trailing '/', delete it
RESULTDIR=${RESULTDIR/%\//}

if [ ! -d $RESULTDIR ] ; then 
   printf "Output directory $RESULTDIR does not exist.\n\n"
   printUsage; exit 1 
fi

# Error check hostname on manual execution of script.
TEMP=$($SSHCOMMAND $HOSTNAME 'exit' 2>&1)
# If the host is not in the 'known hosts' list, this will return a benign 'Permanently added..' error which we can ignore.
if [[ "$TEMP" && ! "$TEMP" =~ "Permanently added" ]]; then
   printf "The Reporter was unable to access $HOSTNAME.\nPlease verify that $HOSTNAME can be resolved from the Prime Client.\n$TEMP\n"
   printUsage; exit 1
fi

# Try finding ESXi host version.
HOSTVERSION=$($SSHCOMMAND $HOSTNAME "vmware -v" 2>&1)  # i.e., VMware ESXi 7.0.3 build-20036589.
# Try finding vCenter Server version.
if [[ ! "$HOSTVERSION" =~ "VMware ESXi" ]]; then
   HOSTVERSION=$($SSHCOMMAND $HOSTNAME "grep 'VMware vCenter' /etc/issue" 2>&1)  # i.e., VMware vCenter Server 7.0.3.00800
fi
if [[ "$HOSTVERSION" =~ "VMware" ]]; then 
   printf "$(date -u): Host $HOSTNAME is running $HOSTVERSION.\n"
else
   printf "The Reporter was unable to determine the ESXi or vCenter Server version of $HOSTNAME.\n$HOSTVERSION\n"
   printUsage; exit 1
fi

# Create working directory which is where all temporary files and activity will happen.
mkdir -pm 755 $WORKDIR
cd $WORKDIR
printf "$(date -u): Starting Configuration Gathering Script for VMware vSphere on $HOSTNAME\n"
printf "$(date -u): $0 $REPORTERVERSION\n"

# Set host type, script, and manifest for ESXi.
if [[ "$HOSTVERSION" =~ "ESXi" ]]; then
   HOSTSCRIPT=Reporter-Host.sh
   MANIFESTFILE=Reporter-Host.mfx
# Set host type, script, and manifest for vCenter Server.
elif [[ "$HOSTVERSION" =~ "vCenter" ]]; then
   HOSTSCRIPT=Reporter-VC.sh
   MANIFESTFILE=Reporter-VC.mfx
fi
# Copy host script to host.
printf "$(date -u): Copying Reporter script to $HOSTNAME\n"
$SCPCOMMAND $SDKDIR/$HOSTSCRIPT $HOSTNAME:/tmp 2>&1 >/dev/null | tee -a $OUTPUT_ERR
$SSHCOMMAND $HOSTNAME "chmod 755 /tmp/$HOSTSCRIPT"  2>>$OUTPUT_ERR 1>>$OUTPUT_WARN
# Copy manifest to host.
printf "$(date -u): Copying manifest file to $HOSTNAME\n"
$SCPCOMMAND $SDKDIR/$MANIFESTFILE $HOSTNAME:/tmp 2>&1 >/dev/null | tee -a $OUTPUT_ERR
# Run reporter script on host with hostname as argument.
# Send all stdout and stderr to temporary files output_err.txt and output_warn.txt
printf "$(date -u): Running Reporter script on $HOSTNAME. This will take some time.\n"
$SSHCOMMAND $HOSTNAME "/tmp/$HOSTSCRIPT $HOSTNAME" 2>>$OUTPUT_ERR 1>>$OUTPUT_WARN
# Remove host script and manifest from host.
$SSHCOMMAND $HOSTNAME "rm -f /tmp/$HOSTSCRIPT /tmp/$MANIFESTFILE" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR

# Transfer standard output file from host.
$SCPCOMMAND $HOSTNAME:/tmp/ReporterFile.txt $WORKDIR 2>&1 >/dev/null | tee -a $OUTPUT_ERR
if [ -e $WORKDIR/ReporterFile.txt ] ; then
   # Remove file from host.
   $SSHCOMMAND $HOSTNAME "rm -f /tmp/ReporterFile.txt" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
else
   # Report error if the transfer failed
   printf "$(date -u): The Reporter did not run correctly on $HOSTNAME.  Please rerun manually.\n"
   echo "The Reporter did not run correctly on $HOSTNAME.  Please rerun manually." | tee -a $OUTPUT_ERR
   exit 1
fi

###############################################################################
# Post-processing
###############################################################################
printf "$(date -u): Configuration-gathering script finished. Beginning post-processing.\n"
REPORTERFILEPATH=$(cat ReporterFile.txt)
printf "$(date -u): Transferring Reporter file from $HOSTNAME to $RESULTDIR\n"
$SCPCOMMAND $HOSTNAME:"$REPORTERFILEPATH" $RESULTDIR >/dev/null # copy reporter file directly into RESULTDIR
VMSUPPORTDIR=$(echo ${REPORTERFILEPATH%\.tar.xz})		# remove .tar.xz from end of filename
$SSHCOMMAND $HOSTNAME "rm -rf $REPORTERFILEPATH $VMSUPPORTDIR" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Remove reporter file from host after completed scp
SUBMITFILE=$(basename $REPORTERFILEPATH)
printf "$(date -u): Post-processing finished. Starting cleanup.\n"

# Set output to "None" if no errors or warnings detected.
if [ "$(cat $OUTPUT_ERR)" = "" ] ; then
   echo "None" > $OUTPUT_ERR
else
   OUTPUT_FAIL=1
fi
if [ "$(cat $OUTPUT_WARN)" = "" ] ; then
   echo "None" > $OUTPUT_WARN
fi

# Publish errors, warnings and location of reporter .tar.xz file for calling scripts & harness
REPORTERRESULTFILE=$RESULTDIR/ReporterResult-$HOSTNAME.txt

printf "$(date -u), Reporter $REPORTERVERSION\n" > $REPORTERRESULTFILE 2>/dev/null
printf "reporter_file:$RESULTDIR/$SUBMITFILE\n" >> $REPORTERRESULTFILE 2>/dev/null
printf "reporter_dir:$HOSTREPORTERDIR\n" >> $REPORTERRESULTFILE 2>/dev/null  # TODO: Remove: $HOSTREPORTERDIR deleted via SSH/rm -rf above, so harness no longer needs to
printf "failed:$OUTPUT_FAIL\n" >> $REPORTERRESULTFILE 2>/dev/null
printf "errors:$(cat $OUTPUT_ERR)\n" >> $REPORTERRESULTFILE 2>/dev/null 
printf "warnings:$(cat $OUTPUT_WARN)\n" >> $REPORTERRESULTFILE 2>/dev/null

# Remove non-ascii characters from $REPORTERRESULTFILE. Non-ascii characters cause STAX to crash.
perl -i -pe 's/[^[:ascii:]]//g' $REPORTERRESULTFILE 2>/dev/null

cd $STARTDIR
# Delete working directory and all temporary files
rm -rf $WORKDIR 2>/dev/null

printf "$(date -u): Cleanup finished.\n"
#printf "$(date -u): Include $RESULTDIR/$SUBMITFILE as part of your benchmark submission.\n"
#printf "$(date -u): View Reporter results at $RESULTDIR/ReporterResult-$HOSTNAME.txt\n"
printf "$(date -u): Done for $HOSTNAME!\n"
