#!/bin/pwsh

#############################################################################
#
# SPECvirt Datacenter 2021
#
# Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC)
# All rights reserved.
#
##############################################################################

# deployVM-Any.ps1 - Deploy any AIO, HammerDB, or BigBench workload VM or Client VM.
#   Assigns macAddress, memory size, vCPUs, storagePool and cluster, adds second disk 
#   if needed and powers on the VM.
#
# pwsh deployVM-Any.ps1 <macAddress> <targetVM> <memoryGB> <vCpu> <storagePool> <cluster>
#
# Get and set variables from Control.config
$configfile = ConvertFrom-StringData (Get-Content /export/home/cp/bin/Control.config -raw)
$ProgressPreference = "SilentlyContinue"

$mgmtServerIP       = $configfile.Get_Item("mgmtServerIP")
$virtUser           = $configfile.Get_Item("virtUser")
$virtPassword       = $configfile.Get_Item("virtPassword")
$templateName       = $configfile.Get_Item("templateName")
$clientTemplateName = $configfile.Get_Item("clientTemplateName")
$clientCluster      = $configfile.Get_Item("clientCluster")
$clientStoragePool  = $configfile.Get_Item("clientStoragePool")
$cluster            = $configfile.Get_Item("cluster")

# Parameters from harness
$macAddress      = [string]$args[0]
$targetVM        = [string]$args[1]
$memoryGB	 = [string]$args[2] 
$vCpu 		 = [string]$args[3]
$storagePool     = [string]$args[4]

$current = Get-Date -Format "ddd MMM dd HH:mm:ss UTC yyyy"   

if ($targetVM -like "svdc-t*-client")
{
	$templateName = $clientTemplateName
}

$diskSizeGB = 0
if ( $targetVM -like "*bbdn?" )
   { $diskSizeGB = 60 }
elseif ( $targetVM -like "*hdb" )
   { $diskSizeGB = 50 }

$memoryGB = 0
if ( $targetVM -like "*bbdn?" -or  $targetVM -like "*hdb" )
   { $memoryGB = 24 }
elseif ( $targetVM -like "*bbdb" ) 
   { $memoryGB = 4 }
else
   { $memoryGB = 8 }

Try
{
   #Write-Host "$current Info: Connect to vSphere"
   $vc = Connect-VIServer -Server $mgmtServerIP -User $virtUser -Password $virtPassword
}
Catch
{
   Write-Host "Error on Connect-VIServer at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
}

$objIs = "TBD"
if ( $vc.IsConnected )
{
     $isTemplate = Get-Template  -Name $templateName -Verbose:$false -ErrorAction Ignore
     if ($isTemplate)
     {
          $objIs = "template"
     }
     else
     {
        $isVM = Get-VM  -Name $templateName -Verbose:$false -ErrorAction Ignore
        if ($isVM)
        {
           $objIs = "VM"
        }
        else
        {
   	   Write-Host "Error: $templateName not a Template or VM  at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
        }
     }

   Try
   {
      if ($objIs -eq "template") {
         #Write-Host "$current Info: Deploying $targetVM from Template: $templateName"
         $VMnew = New-VM -Name $targetVM -Template $isTemplate -ResourcePool $cluster -Datastore $storagePool -confirm:$false | Set-VM -MemoryGB $memoryGB -NumCpu $vCpu -CoresPerSocket $vCpu -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue -Confirm:$False -Verbose:$false
      }
      else 
      {
         #Write-Host "$current Info: Deploying $targetVM from template VM: $templateName"
         $VMnew = New-VM -Name $targetVM -VM $isVM -ResourcePool $cluster -Datastore $storagePool -confirm:$false | Set-VM -MemoryGB $memoryGB -NumCpu $vCpu -CoresPerSocket $vCpu -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue -Confirm:$False -Verbose:$false
      }
   }
   Catch 
   {
      Write-Host "Error creating VM from $objIs : $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" 
      Exit
   }
   if (!$VMnew) {  Write-Host "Error  no VMnew $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)"; Exit }
  
   Try
   {
      #Write-Host "$current Info: Configure network adapter"
      Get-NetworkAdapter -VM $VMnew | Set-NetworkAdapter -MacAddress $macAddress -WakeOnLan:$true -confirm:$false -Verbose:$false | Out-Null
   }
   Catch
   {
      Write-Host "Error setting VM network: $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" 
   }
   
   if ($diskSizeGB) 
   {
         Try
         {
            Write-Host "$current Info: Adding disk to ${targetVM}"
            Get-VM ${targetVM} | New-HardDisk -CapacityGB $diskSizeGB -Persistence persistent -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue -Confirm:$False -Verbose:$false | Out-Null
   }
         Catch
         {
            Write-Host "Error adding Disk at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
         }
   }
}

Disconnect-VIServer -Confirm:$False
