#!/bin/pwsh

#############################################################################
#
# SPECvirt Datacenter 2021
#
# Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC)
# All rights reserved.
#
##############################################################################

# deployVM-ClientToHost.ps1 - Deploy Client VM to given storage pool and host
#
# pwsh deployVM-ClientToHost.ps1 <macAddress> <targetVM> <memoryGB> <vCpu> <storagePool> <hostName>
#
# Get and set variables from Control.config
$configfile = ConvertFrom-StringData (Get-Content /export/home/cp/bin/Control.config -raw)
$ProgressPreference = "SilentlyContinue"

$mgmtServerIP       = $configfile.Get_Item("mgmtServerIP")
$virtUser           = $configfile.Get_Item("virtUser")
$virtPassword       = $configfile.Get_Item("virtPassword")
$clientTemplateName = $configfile.Get_Item("clientTemplateName")
$clientCluster      = $configfile.Get_Item("clientCluster")
$clientStoragePool  = $configfile.Get_Item("clientStoragePool")

# Parameters from harness
$macAddress      = [string]$args[0]
$targetVM        = [string]$args[1]
$memoryGB	 = [string]$args[2] 
$vCpu 		 = [string]$args[3]
$storagePool     = [string]$args[4]
$hostName        = [string]$args[5]

$templateName    = $clientTemplateName
$current         = Get-Date -Format "ddd MMM dd HH:mm:ss UTC yyyy"   

Try
{
   #Write-Host "$current Info: Connect to vSphere"
   $vc = Connect-VIServer -Server $mgmtServerIP -User $virtUser -Password $virtPassword
}
Catch
{
   Write-Host "Error on Connect-VIServer at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
}

$objIs = "TBD"
if ( $vc.IsConnected  -and ($hostName) )
{
     $isTemplate = Get-Template  -Name $templateName -Verbose:$false -ErrorAction Ignore
     if ($isTemplate)
     {
          $objIs = "template"
     }
     else
     {
        $isVM = Get-VM  -Name $templateName -Verbose:$false -ErrorAction Ignore
        if ($isVM)
        {
           $objIs = "VM"
        }
        else
        {
   	   Write-Host "Error: $templateName not a Template or VM  at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
        }
     }

   Try
   {
      if ($objIs -eq "template") {
         Write-Host "$current Info: Deploying $targetVM from Template: $templateName"
         $VMnew = New-VM -Name $targetVM -Template $isTemplate -ResourcePool $hostName -Datastore $storagePool -confirm:$false | Set-VM -MemoryGB $memoryGB -NumCpu $vCpu -CoresPerSocket $vCpu -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue -Confirm:$False
      }
      else 
      {
         Write-Host "$current Info: Deploying $targetVM from template VM: $templateName"
         $VMnew = New-VM -Name $targetVM -VM $isVM -ResourcePool $hostName -Datastore $storagePool -confirm:$false | Set-VM -MemoryGB $memoryGB -NumCpu $vCpu -CoresPerSocket $vCpu -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue -Confirm:$False
      }
   }
   Catch 
   {
      Write-Host "Error creating VM from $objIs : $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" 
      Exit
   }
   if (!$VMnew) {  Write-Host "Error  no VMnew $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)"; Exit }
  
   Try
   {
      #Write-Host "$current Info: Configure network adapter"
      Get-NetworkAdapter -VM $VMnew | Set-NetworkAdapter -MacAddress $macAddress -WakeOnLan:$true -confirm:$false | Out-Null
   }
   Catch
   {
      Write-Host "Error setting VM network: $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" 
   }
   
   Try
   {
      #Write-Host "$current Info: Starting VM"
      Start-VM -VM $VMnew -Confirm:$false -Verbose:$false  -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue | Out-Null
      Write-Host "$current $targetVM deployed"    
 }
   Catch
   {
      Write-Host "Error at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
   }
   
}

Disconnect-VIServer -Confirm:$False
