#!/bin/pwsh

#############################################################################
#
# SPECvirt Datacenter 2021
#
# Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC).
# All rights reserved.
#
##############################################################################

# powerOffAllTiles.ps1 - Power off all VMs on the SUT cluster
#   To shut down client VMs on client cluster specify name of clientCluster as $1
#   Used by: prepTestRun.sh

# # Note: Uses Stop-VMGuest to gracefully shutdown the VM, then
# #  waits up to 4 x #VMs seconds and then attempts to PowerOff
# #  via Stop-VM any VM that had not gracefully shutdown.
#
# optional $args[0] overrides default from Control.config
# optional $args[0] overrides default match string of "svdc-t*"
#
# Get and set variables from Control.config
$configfile = ConvertFrom-StringData (Get-Content /export/home/cp/bin/Control.config -raw)

$mgmtServerIP  = $configfile.Get_Item("mgmtServerIP")
$virtUser      = $configfile.Get_Item("virtUser")
$virtPassword  = $configfile.Get_Item("virtPassword")
$cluster       = $configfile.Get_Item("cluster")

# Get optional override default $cluster name from harness
if ($args.Count -gt 0)
{
   $cluster      = [string]$args[0]
   if ($cluster -eq "clientCluster")
   {
	$cluster = $configfile.Get_Item("clientCluster")
	Write-Host "Override cluster: $cluster"
   }
}

Try
{
   #Write-Host "Info: Connect to vSphere"
   $vc = Connect-VIServer -Server $mgmtServerIP -User $virtUser -Password $virtPassword
}
Catch
{
   Write-Host "Error: Connect-VIServer at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
}

$vmPoweredON=Get-VM -Location $cluster | Where-Object { $_.PowerState -eq 'PoweredOn' -and $_.name -notlike '*client' -and $_.name -like "svdc-t*" }
write-host "Info: VMs powered on: $vmPoweredON"
$CountVMs = $vmPoweredON.count
$firstCount=$CountVMs

if ($CountVMs -gt 0) {

  foreach ($targetVM in $vmPoweredON)
  {
   Try
   {
     Stop-VMGuest -VM $targetVM -Verbose:$false -Confirm:$false -ErrorAction SilentlyContinue -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue | Out-Null
   write-host "Stopping $targetVM"     
   }
     Catch
   {  
     Write-Host "Error shutting down VM : $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" -ForegroundColor Red
   }
  }
  
  Write-Host "Info: Waiting for shutdown to complete"
  start-sleep 3
  $maxSleep = $firstCount * 4
  $vmPoweredON=Get-VM -Location $cluster | Where {$_.PowerState -eq "PoweredOn"} | Where-Object {($_.name -like "svdc-t*")}
  $CountVMs = $vmPoweredON.count
  
  $ctr=0
  while ($CountVMs -ge 1) {
     start-sleep 1
     $ctr++
     $vmPoweredON=Get-VM -Location $cluster | Where {$_.PowerState -eq "PoweredOn"} | Where-Object {($_.name -like "svdc-t*")}
     $CountVMs = $vmPoweredON.count
     Write-Host "Info: VMs to power off: $CountVMs"
     if ($CountVMs -eq 0 -or $ctr -eq $maxSleep) {
     write-host "VMs powered on: $CountVMs" 
     break;
     }
  }
  
  #$vmPoweredON=Get-VM -Location $cluster | Where {$_.PowerState -eq "PoweredOn"} | Where-Object {($_.name -like "svdc-t*")}
  #$CountVMs = $vmPoweredON.count
  #if ($CountVMs -ge 1) { 
  #  Write-Host "Info: Powering Off remaining $CountVMs VMs"  
  #  foreach ($targetVM in $vmPoweredON){
  #    Try
  #    {
  #      Stop-VM -VM $targetVM -Verbose:$false -Confirm:$false -ErrorAction SilentlyContinue -WarningAction:SilentlyContinue -InformationAction:SilentlyContinue | Out-Null
  #    }
  #    Catch
  #    {  
  #      Write-Host "Error shutting down VM : $_.Exception.Message at line $($_.InvocationInfo.ScriptLineNumber)" -ForegroundColor Red
  #    }
  #  }
  #}
}

Disconnect-VIServer -Confirm:$False 
