#!/bin/pwsh

###########################################################################
#                                                                         
#  SPECvirt Datacenter 2021
#
#  Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC). 
#  All rights reserved.                                                   
#                                                                         
###########################################################################

#  sutConfig.ps1: Start/End Run Config script for SPECvirt Datacenter 2021
#
#  This script collects testbed configuration information at 
#  various levels: Datacenter, Cluster, Host, and VM and
#  generates the startRun.config and endRun.config files that
#  is used for validation of a VMware vSphere testbed at the beginning 
#  and end of the test. Additional information is collected by other 
#  means for formal validation of a compliant test.
 
# Get and set variables from Control.config
$configfile = ConvertFrom-StringData (Get-Content /export/home/cp/bin/Control.config -raw)
$ErrorActionPreference = "SilentlyContinue"
$PSStyle.OutputRendering = 'PlainText'

$mgmtServerIP  = $configfile.Get_Item("mgmtServerIP")
$virtUser      = $configfile.Get_Item("virtUser")
$virtPassword  = $configfile.Get_Item("virtPassword")
$cluster       = $configfile.Get_Item("cluster")
$clientCluster = $configfile.Get_Item("clientCluster")

Try
{
   $runFlag       = [string]$args[0]
}
Catch
{
   Write-Host "Error: Select 0 for start test, 1 for end test."
   break
}
Try
{
   $vc = Connect-VIServer -Server $mgmtServerIP -User $virtUser -Password $virtPassword
}
Catch
{
   Write-Host "Error: Connect-VIServer at Line $($_.InvocationInfo.ScriptLineNumber): $_ "
}

# Set counters
$hostCount=0
$offline=0
$online=0
$sut_running=0
$sut_down=0

$benchmark_vmCount_running=0
$benchmark_vmCount_down=0
$migrations=0
$startTime=0
$endTime=0

$dcCount=1
$hostCount=1
$vmCount=1
$clusterCount=1
$storageDomainCount=1

# Get datacenter info
$dc = Get-Datacenter -Cluster $cluster 
Write-Host "SUT.Datacenter.name[$dcCount]=$($dc)"

# Get SUT and client cluster info
$sut = Get-Cluster | Where {($_.Name -eq $cluster)}
Write-Host "SUT.Cluster.name[$clusterCount]=$($sut.Name)"
$clusterCount=$clusterCount+1
$client = Get-Cluster | Where {($_.Name -eq $clientCluster)}
Write-Host "SUT.Cluster.name[$clusterCount]=$($client.Name)"

# Get SUT host info
$vmhosts = Get-Cluster -Name $cluster | Get-VMHost 
foreach($vmhost in $vmhosts) {
  Write-Host "SUT.Host.name[$hostCount]=$($vmhost.Name)"
     if ($vmhost.ConnectionState -eq 'Connected') {
        Write-Host "SUT.Host.state[$hostCount]=up" 
        $online=$online + 1
        } 
     elseif ($vmhost.ConnectionState -eq 'Maintenance') {
        Write-Host "SUT.Host.state[$hostCount]=maintenance" 
        $offline=$offline + 1
        }
  $hostCount=$hostCount+1
}
Write-Host "SUT.Host.offline.Total=$($offline)"
Write-Host "SUT.Host.online.Total=$($online)"

# Get storage info
$storagePools = Get-Cluster -Name $cluster | Get-Datastore | where{$_.ExtensionData.Summary.MultipleHostAccess}   
foreach ($storagePool in $storagePools) {
   Write-Host "SUT.Storage.Pool[$storageDomainCount]=$($storagePool.Name)" 
   $storageDomainCount=$storageDomainCount+1
}

# Get SUT VM info
$vms = Get-VM | Where{$_.Name -match '^svdc-t\d{3}-.*' -and $_.Name -notMatch 'client'}
foreach ($vm in $vms) {                                                                                                                              
  Write-Host "SUT.VM.name[$vmCount]=$($vm.Name)"
     if ($vm.PowerState -eq 'PoweredOn') {
        Write-Host "SUT.VM.state[$vmCount]=up" 
        $benchmark_vmCount_running=$benchmark_vmCount_running+1
        }
     else {
        Write-Host "SUT.VM.state[$vmCount]=down" 
        $benchmark_vmCount_down=$benchmark_vmCount_down+1
        }
  Write-Host "SUT.VM.vCPU[$vmCount]=$($vm.NumCPU)"
  Write-Host "SUT.VM.memory[$vmCount]=$($vm.MemoryMB)"
  #Write-Host "SUT.VM.host[$vmCount]=$($vm.VMhost)"
  $vmCount=$vmCount+1
}
Write-Host "Benchmark.VM.down.total=$($benchmark_vmCount_down)"
Write-Host "Benchmark.VM.running.total=$($benchmark_vmCount_running)"

if ($runFlag -eq '1') {
   $endTime=(Get-Date -format s)
   Write-Host "Benchmark.test.end=$($endTime)"
} else { $startTime=(Get-Date -format s)
   Write-Host "Benchmark.test.start=$($startTime)"
   }

Disconnect-VIServer -Confirm:$False
