#!/bin/bash

# This script will change ethernet device settings and IP adresses
# to adapt to different devices and network topologies.  The default
# device is a eth0 (virtual) with a simple, single network (255.255.0.0)
# for all network devices.
# Eth1 is reserved for pass-through devices.  If eth1 is present, shut
# down eth0 and migrate the network config to eth1.  This is done because
# it is easier to manage the IP address via eth0, since we have better
# control over the MAC and can easily leverage DHCP.
# Eth2 is reserved for special private networks, between a pair of 
# web and infra VM types, and/or a pair of App and DB VM types.
# If eth2 is present, set up IP's on the device and update /etc/hosts
# to ensure the connection to the VM's sibling is through eth2.

if [ -e /sys/class/net/eth1 ]; then
   eth0_ip=`ip addr show dev eth0 | grep "inet " | awk '{print $2}' | awk -F/ '{print $1}'`
   if [ -z "$eth0_ip" ]; then
	echo "no ip address for eth0, exiting"
        exit 1
   fi
   def_route_gw=`ip route show | grep default | awk '{print $3}'`
   echo migrating eth0 ip $eth0_ip to eth2
   ifdown eth0
   ifconfig eth1 down
   ifconfig eth1 $eth0_ip netmask 255.255.0.0 up
   route add -net default gw $def_route_gw dev eth1
fi 

# if there is an eth2 device, set up a IP for a point-to-point network between web&infra and app&db
# remove any old host/ip info first
sed -i /etc/hosts -e '/.*-.*t.*/d' -e '/^$/d'
if [ -e /sys/class/net/eth2 ]; then
    vmtype=`hostname -s | sed -e s/.*-// -e s/t..//`
    tile=`hostname -s | sed -e s/vg-${vmtype}t// -e s/^0*//`
    web_eth2_ip="192.132.${tile}.1"
    infra_eth2_ip="192.132.${tile}.2"
    app_eth2_ip="192.133.${tile}.1"
    db_eth2_ip="192.133.${tile}.2"
    case $vmtype in
        "web")
        sibling_hostname=`hostname -s | sed -e s/web/infra/`
	sibling_eth2_ip=$infra_eth2_ip
	my_eth2_ip=$web_eth2_ip
        ;;
        "infra")
        sibling_hostname=`hostname -s | sed -e s/infra/web/`
	sibling_eth2_ip=$web_eth2_ip
	my_eth2_ip=$infra_eth2_ip
        ;;
        "app")
        sibling_hostname=`hostname -s | sed -e s/app/db/`
	sibling_eth2_ip=$db_eth2_ip
	my_eth2_ip=$app_eth2_ip
        ;;
        "db")
        sibling_hostname=`hostname -s | sed -e s/db/app/`
	sibling_eth2_ip=$app_eth2_ip
	my_eth2_ip=$db_eth2_ip
	;;
    esac
    printf "$sibling_eth2_ip\t\t$sibling_hostname\n" >>/etc/hosts
    ifconfig eth2 $my_eth2_ip netmask 255.255.255.0 up
fi
