/*++

  Copyright (c) 1999 Microsoft Corporation   
  
    Module Name:
    Logfile.CXX

    Created: Neel Jain (njain)
    
    Modified:
    Tai-Yi Huang (tyhuang): changed for SPECweb99/VC++
          
--*/

#include "logfile.hxx"
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>

char DEFLOGFILENAME[] = "PostLog";
struct _logInfo logInfo;

extern   char postLogFormat[];

//= "%10d %10d %10d %5d %2d %2d %10d %-60.60s %10d %10d\n" ;

static int logRecordLen = 139;
static int logRdMember1237910 = 11;
static int logRdMember4 = 6;
static int logRdMember56= 3;
static int logRdMember8 = 61;

//
// Routines to allocate and free logBufs
//

inline void
initLogBuf(logBuf *theBuf, HANDLE fileHandle, unsigned __int64 offset, unsigned bufSpace)

  /*++

Routine Description:

  Initialises a new logBuf structure. Assumes that theBuf has been
  initialised with a properly allocated buffer for log data

Arguments:

  initial values for the logBuf

return Value:

  no return value

--*/
{
    if (theBuf)
    {
        theBuf->logFileOffset = offset;
        theBuf->logFile = fileHandle;
        theBuf->next = NULL;
        theBuf->bytesUsed = 0;
        theBuf->bytesAvailable = bufSpace;
    }
}

logBuf *
newLogBuf(unsigned __int64 offset, HANDLE fileHandle, unsigned int bufferSpace)
/*++

Routine Description:

    Allocates and initialises a new logBuf structure

Arguments:

    initial values for the logBuf

return Value:

    On success, the function returns the address of the new logBuf
    NULL is returned on failure

--*/
{
    logBuf *retBuf;
    char *dataBuf;

    //
    // first try to allocate everything
    //

    retBuf = (logBuf *) malloc(sizeof(logBuf));
    if (!retBuf)
    {
        OutputDebugString("Allocation of a logBuf structure failed\n");
        return (NULL);
    }

    dataBuf = (char *) VirtualAlloc(
                        NULL,
                        logInfo.bufferSize,
                        MEM_RESERVE | MEM_COMMIT,
                        PAGE_READWRITE
                        );

    if (!dataBuf)
    {
        OutputDebugString("Allocation of a log data buffer failed\n");
        free (retBuf);
        return (NULL);
    }

    //
    // Everything's been allocated. Set up the structure
    // First set up the data buffer
    //

    if (bufferSpace > logInfo.bufferSize)
    {
        bufferSpace = logInfo.bufferSize;
    }
    
    retBuf->logData = dataBuf;

    initLogBuf(retBuf, fileHandle, offset, bufferSpace);

    return (retBuf);
}



HANDLE
NewLogFile (void)
/*++

Routine Description:
  
    This function opens a file with the baseName and a count suffix
    
Arguments:
      
    None. Uses globals.

return Value:
        
    The return value from CreateFile
          
--*/
{
    HANDLE newFile; 
	
	DWORD flags = FILE_FLAG_RANDOM_ACCESS;
    
  
	 newFile = CreateFile(logInfo.logFileBaseName,
		GENERIC_READ | GENERIC_WRITE,
		FILE_SHARE_READ,
		NULL,
		CREATE_ALWAYS, // OPEN_EXISTING, // CREATE_ALWAYS,
		//FILE_FLAG_NO_BUFFERING, // not a good idea
		flags,
		NULL);
    
	 if (newFile == INVALID_HANDLE_VALUE) {
		 //sendTextOutput("create post log failed.\n");
		 // error creating post log, need more processing
		return 0;
	 }

    return (newFile);
}


UINT
FlushEntryCounter()
/*++
return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

--*/
{
     OVERLAPPED theOverlapped;
     unsigned long numBytesWritten;
     ULARGE_INTEGER fileOffset;
	 logBuf *aBuffer = logInfo.firstBuffer;
     char    counter[COUNTERFIELDS];

     if (!(
          (logInfo.useLog) &&
          (aBuffer) &&
          (aBuffer->bytesUsed) &&
          (aBuffer->logData) &&
          (aBuffer->logFile) &&
          (aBuffer->logFile != INVALID_HANDLE_VALUE)
          ))
     {
         return (STATUS_ERROR);
     }

     //
     // NKJ: need to check on this
     //

     memcpy(&fileOffset, &(aBuffer->logFileOffset), sizeof(fileOffset));

     //
     // The buffer can be flushed -- all parameters are OK
     //

     memset(&theOverlapped, 0, sizeof(theOverlapped));
     
     theOverlapped.Offset = fileOffset.LowPart;
     theOverlapped.OffsetHigh = fileOffset.HighPart;

     // update the counter field in the buffer

     sprintf(counter, "%10d", logInfo.entryCounter);
     memcpy(aBuffer->logData, counter, COUNTERFIELDS);

     if (!WriteFile(aBuffer->logFile,
                    aBuffer->logData,
                    COUNTERFIELDS,
                    &numBytesWritten,
                    &theOverlapped))
     {
         printf("\nLast error: %d\n", GetLastError());
         return (STATUS_ERROR);
     }

     if (COUNTERFIELDS != numBytesWritten)
     {
         return (STATUS_ERROR);
     }

     return (STATUS_SUCCESS);


}


UINT
FlushLogBuffer(logBuf *aBuffer)
/*++

Routine Description:

    Flushes the current buffer to disk. Assumes that aBuffer will not be cleaned up
    during the call. If there is any doubt about this, this routine should be called with
    the logInfo.logLock held (shared or exclusive). The routine assumes that the buffer may
    be in use, so all volatile data (actually, just bytesUsed) is copied.

    It is the responsibility of the calling routine to ensure that numBytesToWrite is
    appropriate for the WriteFile call (eg, multiple of 

Arguments:

    aBuffer: the log buffer to flush to disk
    numBytesToWrite: the number of bytes to write

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

--*/
{
     OVERLAPPED theOverlapped;
     unsigned long numBytesWritten, numBytesToWrite;
     ULARGE_INTEGER fileOffset;

     if (!(
          (logInfo.useLog) &&
          (aBuffer) &&
          (aBuffer->bytesUsed) &&
          (aBuffer->logData) &&
          (aBuffer->logFile) &&
          (aBuffer->logFile != INVALID_HANDLE_VALUE)
          ))
     {
         return (STATUS_ERROR);
     }

     //
     // NKJ: need to check on this
     //

     memcpy(&fileOffset, &(aBuffer->logFileOffset), sizeof(fileOffset));

     //
     // The buffer can be flushed -- all parameters are OK
     //

     memset(&theOverlapped, 0, sizeof(theOverlapped));
     
     theOverlapped.Offset = fileOffset.LowPart;
     theOverlapped.OffsetHigh = fileOffset.HighPart;

     numBytesToWrite = aBuffer->bytesUsed;

     if (!WriteFile(aBuffer->logFile,
                    aBuffer->logData,
                    numBytesToWrite,
                    &numBytesWritten,
                    &theOverlapped))
     {
         printf("\nLast error: %d\n", GetLastError());
         return (STATUS_ERROR);
     }

     if (numBytesToWrite != numBytesWritten)
     {
         return (STATUS_ERROR);
     }

     return (STATUS_SUCCESS);

}


UINT
FlushDirtyBuffers()
/*++

Routine Description:

    Flushes the a list of dirty buffers to disk. 
    Assumes that the list of buffers needs
    to be reversed to be flushed automatically 
    (performance concern, not correctness
    concern). Assumes the dirty buffers are full. 
    The log lock should be held either exclusive
    or shared when this routine is called.

Arguments:

    None

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

--*/
{
    logBuf *dirtyList, *orderedList, *temp;
    UINT retVal = STATUS_SUCCESS;

    //
    // Check for the case where there's nothing to flush. Note that when the
    // log lock is held exclusive, we won't have any competing threads
    //

    if (!logInfo.dirtyBuffers)
    {
        return (retVal);
    }

    //
    // We have something to flush
    //

    while(1)
    {
        dirtyList = logInfo.dirtyBuffers;
        if (dirtyList == InterlockedCompareExchangePointerPrivate((void **) &(logInfo.dirtyBuffers), NULL, dirtyList))
        {
            break;
        }
    }

    //
    // dirtyList is now a private list. Reverse
    //

    temp = NULL;
    orderedList = NULL;
    while (dirtyList)
    {
        temp = dirtyList->next;
        dirtyList->next = orderedList;
        orderedList = dirtyList;
        dirtyList = temp;
    }

    //
    //  TYHUANG: check if the first buffer is in the dirty list, if so
    //  flush the buffer and take it out from the dirty list such that
    //  it will not be returned to the free buffer list
    //

    if (orderedList == logInfo.firstBuffer) {

        if (FlushLogBuffer(orderedList) != STATUS_SUCCESS) 
            return STATUS_ERROR;

        orderedList = orderedList->next;
    }
    else { // search through the rest of dirty buffer

        dirtyList = orderedList;
        temp = dirtyList->next;

        while (temp) {

            if (temp == logInfo.firstBuffer) {

                if (FlushLogBuffer(temp) != STATUS_SUCCESS) 
                    return STATUS_ERROR;

                dirtyList->next = temp->next;
                break;
            }

            dirtyList = temp;
            temp = dirtyList->next;
        }
    }
    
    //
    // Now flush. Still need to figure out a good way to handle failures
    //

    dirtyList = orderedList;
    while (dirtyList)
    {
        temp = dirtyList;
        dirtyList = dirtyList->next;
        if (FlushLogBuffer(temp) != STATUS_SUCCESS)
        {
            printf("Flush failed for buffer %x\n", temp);
            retVal = STATUS_ERROR;
        }
    }

    //
    // I suspect the Interlocked is not necessary here, but for now, it's here.
    //

    while (temp)
    {
        temp->next = logInfo.freeBuffers;
        if ((temp->next) == InterlockedCompareExchangePointerPrivate((void **) &(logInfo.freeBuffers), orderedList, temp->next))
        {
            break;
        }
    }

    return (retVal);

}


UINT
WriteLogShared(postLogRd logRd)
/*++

Routine Description:

    Makes an entry in the log buffer using the vector of buffers
    passed in dataBufs. The routine does its own locking, so logInfo.logLock
    should not be held when this routine is called.

Arguments:

    dataBufs: array of buffers holding the data to be written to the log buffer
    bufLens: lengths of buffers in data
    numEntries: the number of buffers passed in

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

--*/
{
    long totalLen, startPos;
    long i, retVal = STATUS_ERROR;
    char *startAddress;
    
    totalLen =  logRecordLen;

    acquireDualShared(logInfo.logLock);

    if ((logInfo.currentBuffer))
    {
        //
        // Reserve some space, and make sure we didn't over-run the buffer
        // Increment entryCounter 
		//

		AcquireSpinLock((SpinLock*)&(logInfo.slLock));
			startPos = InterlockedExchangeAdd(
				(LONG *)&((logInfo.currentBuffer)->bytesUsed), totalLen);

        //
        // Check for over-run. If not, just go ahead and log. We have a reference on the
        // state of the log, so then only thing that can change is bytesUsed
        //

        if (startPos + totalLen <= (logInfo.currentBuffer)->bytesAvailable)
        {
            //
            // we found enough space. Now start scribbling
            //

            startAddress = (logInfo.currentBuffer)->logData + startPos;
                
			// get the entry number

			logRd.recordNum =  InterlockedIncrement((LONG *)&logInfo.entryCounter);


			ReleaseSpinLock((SpinLock*)&(logInfo.slLock));

            copyLogRecord(startAddress, logRd, totalLen);

            //
            // done. release lock and return success
            //
            releaseDualShared(logInfo.logLock);

            return STATUS_SUCCESS;
        }

        else
        {
            //
            // didn't find space, so undo the increment and fail
            // Note: may be a good idea to have the first thread to overrun the
            // buffer follow a different code path. Maybe next rev.
            //
			

            InterlockedExchangeAdd(&((logInfo.currentBuffer)->bytesUsed), 
                                    -totalLen);

			ReleaseSpinLock((SpinLock*)&(logInfo.slLock));

            releaseDualShared(logInfo.logLock);
            return (STATUS_ERROR);
        }

    }

    else // (logInfo.currentBuffer) turned out to be NULL
    {
        releaseDualShared(logInfo.logLock);
        return (STATUS_ERROR);
    }
   
}

UINT
WriteLogExclusive(postLogRd logRd)
/*++

Routine Description:
  
    Makes an entry in the log buffer using the vector of buffers
    passed in dataBufs. The routine does its own locking. It takes
    the log lock in exclusive mode, so it can replace (logInfo.currentBuffer) if
    needed. This function is not expected to be called for each log entry --
    just at the end of each buffer. It also starts a new file if needed.
    
Arguments:

    dataBufs: array of buffers holding the data to be written to the log buffer
    bufLens: lengths of buffers in data
    numEntries: the number of buffers passed in

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

--*/
{
    
    long i = 0, retVal = STATUS_ERROR, residualBytes, bytesToWrite;
    long totalLen, newBufSize;
    HANDLE newLog;
    logBuf *cbpt;  
	char *logRecordBuf;

    totalLen = logRecordLen;

	//
	//allocate a buffer for log record
	//

	logRecordBuf = (char *)_alloca(logRecordLen);

	if (NULL == logRecordBuf)
    {
        return (STATUS_ERROR);
    }
    
    //
    // Grab the lock and get to work
    //
    
    acquireDualExclusive(logInfo.logLock);

    cbpt = logInfo.currentBuffer;
    
    if (!cbpt)
    {
        releaseDualExclusive(logInfo.logLock);
        return (STATUS_ERROR - 2);
    }

	//
	//copy log record to the record buffer
	//

	logRd.recordNum =  InterlockedIncrement((LONG *)&logInfo.entryCounter);

	copyLogRecord(logRecordBuf, logRd, totalLen);

    //
    // Note that we have the lock exclusive
    // so we won't need to worry about anything changing
    // First we check that we're not going to overflow the file
    //

    if (cbpt->logFileOffset + cbpt->bytesUsed + totalLen > logInfo.fileSize)
    {
        //
        //  TYHUANG: this should not happen in ISAPI. Just return error
        //

        releaseDualExclusive(logInfo.logLock);
        return (STATUS_ERROR);        
    }

    //
    // Here, we know we will not over-run the log file 
    //  (but can still over-run the buffer)
    //

   
    if ((cbpt->bytesUsed + (long)logRecordLen) <= 
        cbpt->bytesAvailable)
    {
        memcpy(&(cbpt->logData[cbpt->bytesUsed]), logRecordBuf, logRecordLen);
        cbpt->bytesUsed += logRecordLen;
    }
    
    else
    {
        //
        // copy what we can. No point checking if bytesToWrite is 0. Note that the separator
        // has to end up in the new buffer because we know we are at least one short on space
        //
        
        bytesToWrite = cbpt->bytesAvailable - cbpt->bytesUsed;
        memcpy(&(cbpt->logData[cbpt->bytesUsed]), logRecordBuf, bytesToWrite);
        residualBytes = logRecordLen - bytesToWrite;
        cbpt->bytesUsed = cbpt->bytesAvailable;
                    
        //
        // Size the new buffer correctly.
        //
        
        if (cbpt->logFileOffset + cbpt->bytesAvailable + logInfo.bufferSize 
			<= logInfo.fileSize)
        {
            newBufSize = logInfo.bufferSize;
        }
        else
        {
            //
			//  TYHUANG: this should not happen in ISAPI
			//

            releaseDualExclusive(logInfo.logLock);
			return (STATUS_ERROR);
        }

        //
        // Put the current buffer on the dirty list. We have the lock exclusive.
        //

        logInfo.currentBuffer->next = logInfo.dirtyBuffers;
        logInfo.dirtyBuffers = logInfo.currentBuffer;

        //
        // Now get a buffer from the free list
        //

        if ((logInfo.freeBuffers))
        {
            (logInfo.currentBuffer) = (logInfo.freeBuffers);
            (logInfo.freeBuffers) = (logInfo.currentBuffer)->next;
            initLogBuf((logInfo.currentBuffer),
                       logInfo.dirtyBuffers->logFile,
                       logInfo.dirtyBuffers->logFileOffset + 
							logInfo.dirtyBuffers->bytesAvailable,
                       newBufSize);
        }
        
        else
        {
            (logInfo.currentBuffer) = 
				newLogBuf(logInfo.currentBuffer->logFileOffset + 
							logInfo.currentBuffer->bytesAvailable,
						  logInfo.currentBuffer->logFile,
						  newBufSize);
            if (!(logInfo.currentBuffer))
            {
                
                releaseDualExclusive(logInfo.logLock);
                return (STATUS_ERROR - 6);
            }
        }
        
        //
        // one way or the other, we made some space in (logInfo.currentBuffer)
        // write the rest of this entry into the brand new buffer
        //
        
		cbpt = logInfo.currentBuffer;

        memcpy(&(cbpt->logData[cbpt->bytesUsed]), 
					logRecordBuf + bytesToWrite, 
					residualBytes
					);
        cbpt->bytesUsed = residualBytes;
        
    } // if overflow
   
    //
    // Terminate the entry
    //

    cbpt = logInfo.currentBuffer;
    // cbpt->logData[cbpt->bytesUsed - 1] = logInfo.terminator;
    
    //
    // we're done logging
    //
    
    releaseDualExclusive(logInfo.logLock);
    
    retVal = STATUS_SUCCESS;
    

    return (retVal);
}



UINT
FlushCachedData()
/*++
outine Description:

    This function holds shared lock and flushed cached data into disk including
        entryCounter,
        dirtyBuffers

Arguments:

    None

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure
--*/
{
    UINT    retVal = STATUS_SUCCESS;

    acquireDualExclusive(logInfo.logLock);

    // First, write dirty buffer

    if (FlushDirtyBuffers() != STATUS_SUCCESS) 
        retVal = STATUS_ERROR;

    // Second, flush the current buffer

    if (logInfo.firstBuffer &&
        FlushEntryCounter() != STATUS_SUCCESS)
        retVal = STATUS_ERROR;

    releaseDualExclusive(logInfo.logLock);

    // 
    // Done flushing data to file cache. Now flush file cache to disk
    //

    if (!FlushFileBuffers(logInfo.currentLogFile)) 
        retVal = STATUS_ERROR; 

    return retVal;
}


UINT
FlushCachedDataExclusive()
/*++
outine Description:

    This function holds exclusive lock and flushed all cached data into disk including
        entryCounter,
		currentBuffer,
        dirtyBuffers

Arguments:

    None

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure
--*/
{
    UINT    retVal = STATUS_SUCCESS;

    //
    // To maximum concurrency we acquire shared lock here to allow
    // other postlog operations to proceed
    //
    
    acquireDualExclusive(logInfo.logLock);

    // First, write dirty buffer

    if (FlushDirtyBuffers() != STATUS_SUCCESS) 
        retVal = STATUS_ERROR;

    // Second, flush the current buffer
    
    if (logInfo.currentBuffer &&
        FlushLogBuffer(logInfo.currentBuffer) != STATUS_SUCCESS)
        retVal = STATUS_ERROR;
    

    // Third, flush the first buffer

    if (logInfo.firstBuffer &&
        FlushEntryCounter() != STATUS_SUCCESS)
        retVal = STATUS_ERROR;

    // 
    // Done flushing data to file cache. Now flush file cache to disk
    //

    if (!FlushFileBuffers(logInfo.currentLogFile)) 
        retVal = STATUS_ERROR; // to be completed: we need to process the error

    releaseDualExclusive(logInfo.logLock);

    return retVal;
}



UINT
CleanupDefaultLog()
/*++
Routine Description:

    This function cleans up any allocated data structures for the
    logging module. It does no locking of its own.

Arguments:

    None

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

TBD:

    Decide what errors need to be caught and reported
--*/
{
    UINT errVal;
    logBuf *temp;


    //
    // Flush the log buffer, close the log file and free all log buffers
    //

    errVal = FlushCachedDataExclusive();

    if ((logInfo.currentLogFile != NULL) && 
        (logInfo.currentLogFile != INVALID_HANDLE_VALUE))
    {
        CloseHandle(logInfo.currentLogFile);
    }

    if (logInfo.firstBuffer) {
        if (logInfo.firstBuffer == logInfo.currentBuffer)
            logInfo.firstBuffer = NULL;
        else {

            if ((logInfo.firstBuffer)->logData) {
                VirtualFree(
                    (logInfo.firstBuffer)->logData,
                    NULL,
                    MEM_DECOMMIT | MEM_RELEASE
                );
            }      

            free(logInfo.firstBuffer);
        }
    }

    if ((logInfo.currentBuffer))
    {
        if ((logInfo.currentBuffer)->logData)
        {
            VirtualFree(
                (logInfo.currentBuffer)->logData,
                NULL,
                MEM_DECOMMIT | MEM_RELEASE
            );
        }
        free((logInfo.currentBuffer));
    }

    //
    //  Walk the list of free log buffers and free them
    //

    while ((logInfo.freeBuffers) != NULL)
    {
        temp = (logInfo.freeBuffers);
        (logInfo.freeBuffers) = (logInfo.freeBuffers)->next;
        if (temp->logData)
        {
            VirtualFree(
                temp->logData,
                NULL,
                MEM_DECOMMIT | MEM_RELEASE
            );
        }
        free(temp);
    }

    return (errVal);
}

UINT
InitDefaultLog(char *logFileName
    )
/*++

Routine Description:

    Initialises the logging information with the information supplied.
    Does no locking of its own.

Arguments:

    logFileName: name of the file to log to
    bufSize: size of buffers used for logging

return Value:

    On success, the function returns STATUS_SUCCESS
    STATUS_ERROR is returned on failure

TBD:

    Decide what errors need to be caught and reported

--*/
{

    ULONG   bufSize = DEFLOGBUFSIZE;
    unsigned __int64 fileSize = DEFLOGFILESIZE;
    char    counter[COUNTERFIELDS];
    char    *startAddress;

    //
    // Check the parameters passed in. First check to see that we have a
    // basename or a callback function to generate file names
    //

    if (!logFileName) {
        logFileName = DEFLOGFILENAME;
    }

    //
    // Now initialise the globals with the (possibly modified) parameters

    logInfo.logFileBaseName = logFileName;
    logInfo.bufferSize = bufSize;
    logInfo.fileSize = fileSize;

    //
    // Generate a file handle to log to
    //

    logInfo.fileCount = 0;
    logInfo.currentLogFile = NewLogFile();

    if ((logInfo.currentLogFile == NULL) || 
        (logInfo.currentLogFile == INVALID_HANDLE_VALUE))
    {
        return (STATUS_ERROR);
    }

    //
    // We've opened a file. Initialise the spin lock and a log buffer
    //
	// InitSpinLock((SpinLock*)&(logInfo.slResetLock));

	InitSpinLock((SpinLock*)&(logInfo.slLock));
    initDualSpinLock(logInfo.logLock);
    logInfo.dirtyBuffers = logInfo.freeBuffers = NULL;

	

    //
    //  allocate the first buffer
    //

    // logInfo.entryCounter = 0; 

    logInfo.firstBuffer = 
    logInfo.currentBuffer = newLogBuf(0, 
                                    logInfo.currentLogFile, 
                                    logInfo.bufferSize
                                    );

    if (!logInfo.currentBuffer)
    {
        CleanupDefaultLog();

        return (STATUS_ERROR);
    }

    logInfo.separator = ' ';
    logInfo.terminator = '\n';
    logInfo.useLog = TRUE;

    //
    // initialize the first buffer
    //

    sprintf(counter, "%10d", 0);
    startAddress = (logInfo.currentBuffer)->logData;

    memcpy(startAddress, counter, COUNTERFIELDS);
    startAddress += COUNTERFIELDS;
    *startAddress = logInfo.terminator;
    (logInfo.currentBuffer)->bytesUsed += COUNTERFIELDS + 1;

	InterlockedExchange((LONG *)&logInfo.entryCounter, 0);
    return (STATUS_SUCCESS);
}


//
//This function turns int to chars with no reverse operation.
//Note, to use this fuction safely, the length of the int should less or equal to the buffer length.
//

void intToCharsNoReverse(char *pBuf, unsigned int inData)
{
	char *pTem, cTem;
	int  iTem;


	if(inData < 10)
	{
		*pBuf = '0' + inData;
		return;
	}

	pTem = pBuf;

	// transfer digital to char from right to left
	while( inData > 0)
	{
		iTem = inData/10;

		*pTem = '0' + (inData - iTem*10);

		inData = iTem;
		pTem--;
	}

	return;
}

//
//This function copy log record into a buffer.
//

UINT copyLogRecord(char *pBuf, postLogRd logRd, int recordLen)
{
	char *pCpyPoint, *pID1;

		//check input parameters
	//if(logRd.pID <= 0)
	//{
	//	return(STATUS_ERROR);
	//}
	
	// set the buffer to ' '
	memset(pBuf,' ', recordLen);

	// memcpy(pBuf, "\n", 1);

	//copy first member for the record to the buffer
	

	pCpyPoint = pBuf + logRdMember1237910-2;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.recordNum));

	//copy second member for the record to the buffer


	pCpyPoint = pCpyPoint + logRdMember1237910;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.timeStamp));

	//copy third member for the record to the buffer

	pID1 = pCpyPoint + 2;

	pCpyPoint = pCpyPoint + logRdMember1237910;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.pID));

	//copy fourth member for the record to the buffer

	pCpyPoint = pCpyPoint + logRdMember4;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.dirNum));

	//copy fifth member for the record to the buffer

	pCpyPoint = pCpyPoint + logRdMember56;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.classNum));

	//copy sixth member for the record to the buffer

	pCpyPoint = pCpyPoint + logRdMember56;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.fileNum));

	//copy seventh member for the record to the buffer

	pCpyPoint = pCpyPoint + logRdMember1237910;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.clientNum));

	//copy eighth member for the record to the buffer

	pCpyPoint = pCpyPoint + 2;
	
	memcpy(pCpyPoint, logRd.fileName, strlen(logRd.fileName));

	//copy nineth member for the record to the buffer

	pCpyPoint = pCpyPoint + logRdMember8;

	memcpy(pCpyPoint, pID1, logRdMember1237910 );

	// pCpyPoint = pCpyPoint  - 2 + logRdMember1237910;
	
	//intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.pID));

	//copy tenth member for the record to the buffer

	pCpyPoint = pCpyPoint - 2 + logRdMember1237910 + logRdMember1237910;
	
	intToCharsNoReverse(pCpyPoint, (unsigned int)(logRd.myCookie));

	//copy record seperator

	pCpyPoint = pCpyPoint + 1;
	
	memcpy(pCpyPoint, "\n", 1);

	return(STATUS_SUCCESS);

}

