/* (C) 1997 Standard Performance Evaluation Corporation */
/*
 *  Time functions
 *  
 *  void spec_msleep(int time);
 *      This will cause this thread/process to sleep for 'time'
 *      milliseconds.
 *
 *  double gettimems(void);
 *      This should return the number of milliseconds that have passed
 *      since getepoch was called.  It returns a double to handle
 *      operating systems that have clock resolutions greater than 1ms.
 *
 *  void gettime(TIMESTORAGE *a);
 *      This fetches the time and stores it in the OS native format.
 *
 *  void getepoch(TIMESTORAGE *a);
 *      This must be the first time-related function called.  It stores
 *      the epoch in 'a'.  'a' may be a NULL to initialize the time
 *      system without storing the value.
 *
 *  int timeval_cmp(TIMESTORAGE *a, TIMESTORAGE *b);
 *      strcmp analogue.  returns -1, 0, or 1 depending on if a > b, 
 *      a == b, or a < b;
 *
 *  int timeval_add(TIMESTORAGE *a, TIMESTORAGE *b);
 *      adds b to a, equivalant to a += b;
 *
 *  int timeval_sub(TIMESTORAGE *a, TIMESTORAGE *b);
 *      subtract b from a, equivalant to a -= b;
 *
 *  long timeval_elapsed(TIMESTORAGE *a, TIMESTORAGE *b);
 *      returns the number of milliseconds elapsed (a-b);
 *
 *  long timeval_zero(TIMESTORAGE *ptr);
 *      set the time to 0.  What 0 means is OS dependant
 *
 *
 *  Miscellaneous 
 *
 *  int set_process_group(void);
 *      Make this process a process group leader if possible.
 *
 *  int strcasecmp(const char *a, const char *b);
 *      case insensitive string compare for those who don't have it
 *
 *
 *  All thread/process abstractions are handled with void pointers.  Be
 *  sure to call the appropriate Destroy routine for each Create that
 *  you perform.
 *
 *  int InitChildLayer(void);
 *      This routine does all of the necessary initialization for this
 *      process so that it can handle child processes/threads.
 *
 *  void CleanupChildLayer(void);
 *      This function is not external.  It is called through atexit.
 *
 *  void *CreateChild(void (*func)(void *), void *data);
 *      Makes a child running function func, data is passed as the input
 *      to the child.
 *
 *  void CancelChild(void *child);
 *      Abort child if possible.  May wait for synchronization point,
 *      may not.  Frees storage for child.
 *
 *  void *MutexCreate(void);
 *      Create a Mutex to lock things with.  Starts off unlocked.
 *
 *  void MutexLock(void *ptr);
 *      Lock the mutex
 *
 *  void MutexUnlock(void *ptr);
 *      Unlock mutex.
 *
 *  void MutexDestroy(void *ptr);
 *      Free mutex and associated storage.
 *
 *
 *  Due to the non-shared memory implementation to access the shared
 *  memory you must Lock the region if you have a Mutex, and you
 *  should.  Then "Read" the region of the memory that you are
 *  interested in.  Modify and Write if you want to change it.  Then
 *  unlock the region.  This is due to the fact that in the worst
 *  case we implement the "shared memory" with files.
 *
 *  void *SHMCreate(size_t size);
 *      Create a "shared memory" region.  
 *
 *  void SHMRead(void *ptr, void *buf, size_t offset, size_t size);
 *      Read a bit of the shared memory in.
 *
 *  void SHMWrite(void *ptr, void *buf, size_t offset, size_t size);
 *      Write some shared memory out.
 *
 *  void SHMDestroy(void *ptr);
 *      Destroy Shared Memory Region
 *
 *  
 *  Thread Specific Pointers.  These are pointers are a way of implementing
 *  unique global data for each thread.
 *
 *  void *TSPCreate(void);
 *      Create a Thread Specific Pointer
 *
 *  int TSPSet(void *tsp, void *data);
 *      Sets the value of the Pointer 'tsp' to 'data'
 *      
 *  void *TSPGet(void *tsp);
 *      Fetch the current value of the tsp.
 *
 *  void TSPDestroy(void *tsp);
 *      Destroy the Pointer.
 */

#include "client.h"

static char *abstract_c_rcsid = "$Id: abstract.c,v 1.8 1998/03/04 07:25:58 channui Exp $";

char **progname;

#if defined(USE_POSIX_THREADS)
    pthread_t main_thread;

int InitChildLayer(void) {
    SUB(InitChildLayer, enter);
#if defined(HAVE_PTHREAD_INIT)
    pthread_init();
#endif
    main_thread=pthread_self();
    return 0;
}

void *CreateChild(void (*func)(void *), void *data) {
    int rc;
    pthread_t *ptr;
    pthread_attr_t attr;

    SUB(CreateChild, enter);

    ptr = (pthread_t *)malloc(sizeof(*ptr));
    if (ptr == NULL) die ("CreateChild: failed to allocate %d bytes\n", 
	                  sizeof(*ptr));

    pthread_attr_init(&attr); 
#ifdef SYSTEM_SCOPE_THREADS
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM); 
#endif
    rc = pthread_create(ptr, &attr, (void *(*)(void*))func, data);
    if (rc) die ("pthread_create: %s\n", strerror(rc));

    rc = pthread_detach(*ptr);
    if (rc) warn ("pthread_detach failed: %s\n", strerror(rc));

    (void) pthread_attr_destroy(&attr);

    return ptr;
}

void CancelChild(void *child) {
    SUB(CancelChild, enter);
    pthread_cancel(*(pthread_t *)child);
    free(child);
}

void *MutexCreate(void) {
    int rc;
    pthread_mutex_t *ptr = (pthread_mutex_t *)malloc(sizeof(pthread_mutex_t));

    SUB(MutexCreate, enter);

    if (ptr == NULL) die ("CreateMutex: failed to allocate %d bytes\n", 
	                  sizeof(pthread_mutex_t));
    if ((rc = pthread_mutex_init(ptr, NULL)))
	die ("MutexCreate: pthread_mutex_init: %s\n", strerror(rc));
    return (void *)ptr;
}

void MutexLock(void *ptr) {
    int rc;

    SUB(MutexLock, enter);
    if ((rc = pthread_mutex_lock((pthread_mutex_t *)ptr)))
	die ("MutexLock: pthread_mutex_lock: %s\n", strerror(rc));
}

void MutexUnlock(void *ptr) {
    int rc;

    SUB(MutexUnlock, enter);
    if ((rc = pthread_mutex_unlock((pthread_mutex_t *)ptr)))
	die ("MutexUnlock: pthread_mutex_unlock: %s\n", strerror(rc));
}

void MutexDestroy(void *ptr) {
    int rc;
    SUB(MutexDestroy, enter);
    if ((rc = pthread_mutex_destroy((pthread_mutex_t *)ptr)))
	die ("MutexDestroy: pthread_mutex_destroy: %s\n", strerror(rc));
    free(ptr);
}

/* Thread Specific Pointers */
void *TSPCreate(void) {
    pthread_key_t *key = malloc(sizeof(*key));
    if (key == NULL) die ("TSPCreate: failed to allocate memory for key\n");
    pthread_key_create(key, NULL);
    TSPSet(key, NULL);
    return key;
}
int TSPSet(void *tsp, void *data) {
    pthread_setspecific(*(pthread_key_t *)tsp, data);
    return 0;
}
void *TSPGet(void *tsp) {
    return pthread_getspecific(*(pthread_key_t *)tsp);
}
void TSPDestroy(void *tsp) {
    pthread_setspecific(*(pthread_key_t *)tsp, NULL);
    pthread_key_delete(*(pthread_key_t *)tsp);
}

#elif defined(_WIN32)
    int main_thread;

void CleanupChildLayer(void) {
    SUB(CleanupChildLayer, enter);
    WSACleanup();
}

int InitChildLayer(void) {
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    int major, minor;

    SUB(InitChildLayer, enter);

    wVersionRequested = MAKEWORD( 2, 0 );

    err = WSAStartup( wVersionRequested, &wsaData );
    if ( err != 0 ) {
	/* Tell the user that we couldn't find a useable */
	/* WinSock DLL.                                  */
	die ("Couldn't find a WinSock DLL!\n");
    }

    /* Confirm that the WinSock DLL supports 2.0.        */
    /* Note that if the DLL supports versions greater    */
    /* than 2.0 in addition to 2.0, it will still return */
    /* 2.0 in wVersion since that is the version we      */
    /* requested.                                        */

    major = LOBYTE( wsaData.wVersion );
    minor = LOBYTE( wsaData.wVersion );

    if ( major < 1 || ( major == 1 && minor < 1 ) ) {
	/* Tell the user that we couldn't find a useable */
	/* WinSock DLL.                                  */
	WSACleanup( );
	die ("Require at least WinSock 1.1, your version is %d.%d!\n", 
	     major, minor);
    }
    return 0;
}

void *CreateChild(void (*func)(void *), void *data) {
    HANDLE rv;
    DWORD id;

    SUB(CreateChild, enter);

    rv = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)func, 
	    data, 0, &id);
    if (rv == NULL)
	    die ("CreateThread failed: error code is %d\n", GetLastError());

    return rv;
}

void CancelChild(void *child) {
    SUB(CancelChild, enter);
    TerminateThread(*(HANDLE *)child, 0);
    free(child);
}

void *MutexCreate(void) {
    HANDLE *ptr, rv;

    SUB(MutexCreate, enter);
	
    ptr = (HANDLE *)malloc(sizeof(*ptr));
    if (ptr == NULL) die ("CreateChild: failed to allocate %d bytes\n", 
	                  sizeof(*ptr));

    rv = CreateMutex(NULL, 0, NULL);
    if (rv == NULL)
	die ("CreateMutex failed\n");

    *(HANDLE *)ptr = rv;
    return ptr;
}

void MutexLock(void *ptr) {
    SUB(MutexLock, enter);
    if (WaitForSingleObject(*(HANDLE *)ptr, INFINITE) != WAIT_OBJECT_0)
	die ("WaitForSingleObject failed\n");
}

void MutexUnlock(void *ptr) {
    SUB(MutexUnlock, enter);
    if (!ReleaseMutex(*(HANDLE *)ptr))
	die ("ReleaseMutex failed\n");
}

void MutexDestroy(void *ptr) {
    SUB(MutexDestroy, enter);
    if (!CloseHandle(*(HANDLE *)ptr))
	die ("CloseHandle failed\n");
    free(ptr);
}

/* 
 * Thread Specific Pointers:
 *
 * Thread-local storage uses the "#define Thread __declspec(thread)"
 * attribute under WIN32.  So the functions
 *
 *		TSPCreate()
 *		TSPSet()
 *		TSPGet()
 *		TSPDestroy() 
 *
 * are unnecessary.  These functions are #defined as do-nothing macros 
 * in abstract.h
 */

/* 
 * Shared Memory Allocation:
 *
 * Since WIN32 uses threads, any memory can be shared so we don't 
 * need special versions of the functions:
 *
 *		SHMdump()
 *		SHMinit()
 *		SHMmalloc()
 *		SHMfree()
 *              SHMdestroy()
 *
 * These functions are #defined in abstract.h to the equivalent malloc() 
 * and free().
 */

#elif defined(USE_SYSV_IPC)
    int main_thread;

void CleanupChildLayer(void) {
    SUB(CleanupChildLayer, enter);
    if (main_thread == getpid()) {
	kill (0, SIGTERM);
    }
}

int InitChildLayer(void) {
    int rc;

    main_thread=getpid();
    SUB(InitChildLayer, enter);

    set_process_group(); /* Make ourselves a process group leader */
    atexit(CleanupChildLayer);
    return 0;
}

void *CreateChild(void (*func)(void *), void *data) {
    int *ptr;
    int pid;

    SUB(CreateChild, enter);

    ptr = (int *)malloc(sizeof(*ptr));
    if (ptr == NULL) die ("CreateChild: failed to allocate %d bytes\n", 
	                  sizeof(*ptr));

    if ((pid = fork()) < 0) { /* error */
	die ("Can't fork: %s\n", strerror(errno));
    } else if (pid) { /* parent */
    } else { /* child */
	*progname = "driver";
	/* Close any files we may have opened */
#if defined(NICE_CHILDREN)
	nice(1);
#endif
	func(data);
	exit (0);
    }

    *ptr = pid;
    return ptr;
}

void CancelChild(void *child) {
    SUB(CancelChild, enter);
    kill(*(int *)child, SIGTERM);
    free(child);
}

void *MutexCreate(void) {
    int *ptr;

    SUB(MutexCreate, enter);

    ptr = (int *)malloc(sizeof(*ptr));
    if (ptr == NULL) die ("CreateChild: failed to allocate %d bytes\n", 
	                  sizeof(*ptr));
    *ptr = sem_create(IPC_PRIVATE, 1);
    if (*ptr == -1) {
	free(ptr);
	ptr = NULL;
    }
LOG (LOG_TRACE,"MutexCreate() = %08x\n", ptr);

    return ptr;
}

void MutexLock(void *ptr) {
    SUB(MutexLock, enter);
LOG (LOG_TRACE,"MutexLock(%08x)\n", ptr);
    sem_wait(*(int *)ptr);
}

void MutexUnlock(void *ptr) {
    SUB(MutexUnlock, enter);
LOG (LOG_TRACE,"MutexUnlock(%08x)\n", ptr);
    sem_signal(*(int *)ptr);
}

void MutexDestroy(void *ptr) {
    SUB(MutexDestroy, enter);
LOG (LOG_TRACE,"MutexDestroy(%08x)\n", ptr);
    sem_rm(*(int *)ptr);
    free(ptr);
}

/* The pointer we return is actually a structure containing the shared memory
   id and a pointer to the shmat-ed region. */

typedef struct local_shm_s {
    int shmid;
    void *buffer;
} local_shm_t;

/* Thread Specific Pointers, we just return a pointer, we use fork/exec
   so our process have their own proces space. */
void *TSPCreate(void) {
    void **key = malloc(sizeof(*key));
    if (key == NULL) die ("TSPCreate: failed to allocate memory for key\n");
    *key = NULL;
    return key;
}
int TSPSet(void *tsp, void *data) {
    *(void **)tsp = data;
    return 0;
}
void *TSPGet(void *tsp) {
    return *(void **)tsp;
}
void TSPDestroy(void *tsp) {
    free(tsp);
}

#else
#error Need to specify type of system you are running with!
#endif

/*****************************************************************************/
/*                              Memory Functions                             */
/*****************************************************************************/

#if defined(USE_SYSV_IPC)

/* These are *very* basic malloc routines, probably want to replace them with
 * something a bit more advanced */

#define SHMB_FREE  (1<<0)
#define SHMB_LAST  (1<<1)
#define SHMB_FIRST (1<<2)

typedef struct shmblock_s {
    int guard1;
    int size;
    int prev_size;
    int flags;
    int guard2;
#ifdef SHMMALLOC_DEBUG
    char file[64];
    int  line;
#endif
} shmblock_t;

void SHMdump(SHM_t *SHM) {
#ifdef SHMMALLOC_PRINT
    shmblock_t *block = SHM->buffer;
    int count = 0;
    if (SHM == NULL) {
	warn ("SHM NULL!\n");
	return;
    }
    MutexLock(SHM->lock);
    do {
	if (block->guard1 != 0x12345678 && block->guard2 != 0x12345678) {
	    warn ("SHMmalloc: guard bytes invalid!\n");
	} else if (block->guard1 != 0x12345678) {
	    warn ("SHMmalloc: guard byte 1 invalid!\n");
	} else if (block->guard2 != 0x12345678) {
	    warn ("SHMmalloc: guard byte 2 invalid!\n");
	}
	LOG(LOG_TRACE, "block %d: %08x(%08x), size=%d, psize=%d, flags=", 
		count++, block, (char *)block+sizeof(*block), block->size, block->prev_size);
	if (block->flags & SHMB_FIRST) LOG(LOG_TRACE, "FIRST ");
	if (block->flags & SHMB_LAST) LOG(LOG_TRACE, "LAST ");
	if (block->flags & SHMB_FREE) LOG(LOG_TRACE, "FREE ");
#ifdef SHMMALLOC_DEBUG
	LOG(LOG_TRACE, "(%s %d)", block->file, block->line);
#endif
	LOG(LOG_TRACE, "\n");
	if (block->flags & SHMB_LAST) {
	    MutexUnlock(SHM->lock);
	    return;
	}
	block = (shmblock_t *)((char *)block + block->size);
    } while (1);
#endif
}

SHM_t *SHMinit(long size) {
    int shmid;
    SHM_t *SHM;
    shmblock_t *block;

    warn("SHMinit(%d)\n", size);

    if ((SHM = calloc(sizeof(*SHM),1)) == NULL)
	die("SHMCreate: calloc error: %s(%d)", strerror(errno), errno);
    if ( (SHM->shmid = shmget(IPC_PRIVATE, size, 0644)) < 0)
	die("SHMCreate: shmget error: %s(%d)", strerror(errno), errno);
    if ( (SHM->buffer = (shmblock_t *)shmat(SHM->shmid, 0, 0)) == (void *) -1)
	die("SHMCreate: shmat error: %s(%d)", strerror(errno), errno);
    SHM->lock = MutexCreate();
    warn("SHM->lock=%08x\n", SHM->lock);
    block = SHM->buffer;
    memset(block, 0, sizeof(*block));
    block->guard1 = block->guard2 = 0x12345678;
    block->size  = size;
    block->flags = SHMB_FREE | SHMB_LAST | SHMB_FIRST;
    return SHM;
}

/* ALIGN_PAGE8(P):  address P at 8-bytes alignment.
   WARNING: any "long double" type will cause SIGBUS core dump as it needs
            32-bytes alignment.
*/
#define ALIGN_PAGE8(P) ((long )(((long)(P) + 7) & ~7))
#define BlockSize  ALIGN_PAGE8(sizeof(shmblock_t))

#ifdef SHMMALLOC_DEBUG
#undef SHMmalloc
void *SHMmalloc(SHM_t *SHM, long size, char *file, int line) {
#else
void *SHMmalloc(SHM_t *SHM, long size) {
#endif
    shmblock_t *block = SHM->buffer;
    MutexLock(SHM->lock);
#ifdef SHMMALLOC_PRINT
LOG (LOG_TRACE, "SHMmalloc(%08x, %d) = ", SHM, size);
#endif
    size = ALIGN_PAGE8(size) + BlockSize;
    do {
	if (block->guard1 != 0x12345678 && block->guard2 != 0x12345678) {
	    warn ("SHMmalloc: guard bytes invalid!\n");
	} else if (block->guard1 != 0x12345678) {
	    warn ("SHMmalloc: guard byte 1 invalid!\n");
	} else if (block->guard2 != 0x12345678) {
	    warn ("SHMmalloc: guard byte 2 invalid!\n");
	}
	if (!(block->flags & SHMB_FREE)) {
	    /* Not free skip this block */
	} else if (block->size > size + BlockSize) {
	    shmblock_t *next_block;

	    next_block            = (shmblock_t *)((char *)block + size);
	    next_block->size      = block->size - size;
	    next_block->prev_size = size;
	    next_block->flags     = block->flags & ~SHMB_FIRST;
	    next_block->guard1    = next_block->guard2 = 0x12345678;

	    block->size           = size;
	    block->flags         &= ~(SHMB_FREE|SHMB_LAST);

#ifdef SHMMALLOC_DEBUG
	    strcpy(block->file, file);
	    block->line = line;
#endif

#ifdef SHMMALLOC_PRINT
LOG (LOG_TRACE, "%08x\n", ((char *)block)+BlockSize);
#endif
	    MutexUnlock(SHM->lock);
	    return ((char *)block)+BlockSize;
	} else if (block->size >= size) {
	    /* Exact size */
	    block->flags &= ~SHMB_FREE;

#ifdef SHMMALLOC_DEBUG
	    strcpy(block->file, file);
	    block->line = line;
#endif

#ifdef SHMMALLOC_PRINT
LOG (LOG_TRACE, "%08x\n", ((char *)block)+BlockSize);
#endif
	    MutexUnlock(SHM->lock);
	    return ((char *)block)+BlockSize;
	}
	if (block->flags & SHMB_LAST) {
#ifdef SHMMALLOC_PRINT
LOG (LOG_TRACE, "(NULL)\n");
#endif
	    MutexUnlock(SHM->lock);
	    return NULL;
	}
	block = (shmblock_t *)((char *)block + block->size);
    } while (1);
}


void  SHMfree  (SHM_t *SHM, void *ptr) {
    shmblock_t *block = (shmblock_t *)((char *)ptr - BlockSize);
    shmblock_t *next_block;
#ifdef SHMMALLOC_PRINT
    LOG(LOG_TRACE, "SHMfree(%08x, %08x)\n", SHM, ptr);
#endif
    MutexLock(SHM->lock);
    block->flags |= SHMB_FREE;
    if (block->guard1 != 0x12345678 && block->guard2 != 0x12345678) {
	warn ("SHMfree: guard bytes invalid!\n");
    } else if (block->guard1 != 0x12345678) {
	warn ("SHMfree: guard byte 1 invalid!\n");
    } else if (block->guard2 != 0x12345678) {
	warn ("SHMfree: guard byte 2 invalid!\n");
    }
    next_block = (shmblock_t *)((char *)block + block->size);
    while (!(block->flags & SHMB_LAST) && (next_block->flags & SHMB_FREE)) {
	block->flags |= next_block->flags;
	block->size  += next_block->size;
	next_block = (shmblock_t *)((char *)block + block->size);
    }
    next_block = (shmblock_t *)((char *)block-block->prev_size);
    if (!(block->flags & SHMB_FIRST) && (next_block->flags & SHMB_FREE)) {
	next_block->flags |= block->flags;
	next_block->size  += block->size;
	block = next_block;
	next_block = (shmblock_t *)((char *)block-block->prev_size);
    }
    MutexUnlock(SHM->lock);
}
void  SHMdestroy  (SHM_t *SHM) {
    SUB(SHMdestroy, enter);
    SHMdump(SHM);
    MutexDestroy(SHM->lock);
    if (shmdt(SHM->buffer) < 0) {
	die("SHMdestroy: shmdt error: %s(%d)", strerror(errno), errno);
    }
    if (shmctl(SHM->shmid, IPC_RMID, 0) < 0)
	die("SHMCreate: shmctl error: %s(%d)", strerror(errno), errno);
    free(SHM);
}

#elif defined(USE_POSIX_THREADS)
void SHMdump(SHM_t *SHM) {
    return;
}

SHM_t *SHMinit(long size) {
    return NULL;
}

#ifdef SHMMALLOC_DEBUG
#undef SHMmalloc
void *SHMmalloc(SHM_t *SHM, long size, char *file, int line) {
#else
void *SHMmalloc(SHM_t *SHM, long size) {
#endif

    return malloc(size);
}


void  SHMfree  (SHM_t *SHM, void *ptr) {
    free(ptr);
}

void  SHMdestroy  (SHM_t *SHM) {
    free(SHM);
}

#endif

/*****************************************************************************/
/*                               Time Functions                              */
/*****************************************************************************/

/*
 * All time is handled in milliseconds.  Therefore in a signed 32 bit integer
 * we can hold 24 days worth of time from an epoch.
 *
 * To use the time functions portably you must first call getepoch before
 * calling any other time function.
 */

#if defined(_WIN32)
void spec_msleep(int time) {
    SUB(spec_msleep, enter);
    Sleep(time);
}

/* 
 * WIN32 returns time in seconds, the only generic high resolution timer
 * available is the GetTickCount which returns the current "tick" in some
 * resolution >= 1ms.  This value will wrap when it overflows.  
 *
 * We are now using QueryPerformanceCounter.  This code has *NEVER BEEN
 * TESTED*.  Not even compiled.
 */

static LARGE_INTEGER Freq;

double gettimems(void) {
    TIMESTORAGE a;

    SUB(gettimems_WIN32, enter);
    gettime(&a);
    /* MC: fixed bug in next line that was causing underflow */
    return (double)(a.QuadPart / (Freq.QuadPart / 1000.0));
}

#define MAX_EPOCH _I64_MAX
static TIMESTORAGE epoch_time;
void gettime(TIMESTORAGE *a) {
    TIMESTORAGE tmp;

    SUB(gettime, enter);
    QueryPerformanceCounter (&tmp);
    if (tmp.QuadPart < epoch_time.QuadPart) { /* Handle overflow case */
	die ("Counter overflowed.  Don't know what the maximum value for a LARGE_INTEGER is\n");
	a->QuadPart = MAX_EPOCH - epoch_time.QuadPart + tmp.QuadPart;
    } else {
	a->QuadPart = tmp.QuadPart - epoch_time.QuadPart;
    }
}
void getepoch(TIMESTORAGE *a) {
    SUB(getepoch, enter);

    if (QueryPerformanceFrequency( &Freq ) == FALSE) {
	die("This machine does not support QueryPerformanceCounter\n");
    }
    QueryPerformanceCounter (&epoch_time);
    if (a != NULL)
	a->QuadPart = 0;
}

/*  b - a ... <=> operator */
int timeval_cmp(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_cmp, enter);
    if (a->QuadPart < b->QuadPart)
	return -1;
    if (a->QuadPart > b->QuadPart)
	return  1;
    return 0;
}

/* add b to a, equivalant to a += b */
int timeval_add(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_add, enter);
    a->QuadPart += b->QuadPart;
    return 0;
}

/* subtract b fromo a, equivalant to a = a - b */
int timeval_sub(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_sub, enter);
    a->QuadPart -= b->QuadPart;
    return 0;
}

/* MC: difference in milliseconds */
long timeval_elapsed(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_elapsed, enter);
    /* MC: fixed an underflow bug here */
    /* MC: return (long)(((a->QuadPart - b->QuadPart) * 1000) / Freq.QuadPart); */
    return (long)((a->QuadPart - b->QuadPart) / (Freq.QuadPart / 1000));

}

/* a = 0 */
void timeval_zero(TIMESTORAGE *ptr) {
    SUB(timeval_zero, enter);
    ptr->QuadPart = 0;
}

#else

#if defined(HAVE_NANOSLEEP) 
void spec_msleep(int time) {

    struct timespec interval, remainder;

    SUB(spec_msleep_nano, enter);
    
#if defined(SLOW_SLEEP)
    /* patch for lazy sleepers */
    if ((time-=9) < 1) return;
#endif

    interval.tv_sec = time/1000;
    interval.tv_nsec = (time%1000)*1000000;

    LOG(LOG_TRACE,"sleep for %d.%09d sec\n", interval.tv_sec, interval.tv_nsec);

    if (nanosleep(&interval, &remainder) == -1) {
        if (errno == EINTR) {
           LOG(LOG_TRACE|LOG_STDERR,"nanosleep interrupted\n");
           LOG(LOG_TRACE|LOG_STDERR,"Remaining secs: %d\n", remainder.tv_sec);
           LOG(LOG_TRACE|LOG_STDERR,"Remaining nsecs: %d\n", remainder.tv_nsec);
         }
        else warn ("Error in nanosleep: %s(%d)\n", strerror(errno), errno);
    }
}
#elif defined(HAVE_USLEEP)
void spec_msleep(int time) {
    int rc;
    SUB(spec_msleep, enter);
#if defined(SLOW_SLEEP)
    /* patch for lazy sleepers */
    if ((time-=9) < 1) return;
#endif
    sleep(time/1000);
    usleep((time%1000)*1000);
}
#else
/* If we don't have usleep emulate it with select, this doesn't have to be
   this complicated, as it handles interrupts which we shouldn't get, and
   the non-select version doesn't handle anyway. */
void spec_msleep(int time) {
    TIMESTORAGE begin, end;
    struct timeval tv;
    long remain;

    SUB(spec_msleep, enter);

#if defined(SLOW_SLEEP)
    /* patch for lazy sleepers */
    if ((time-=9) < 1) return;
#endif
    gettime(&begin);
    while (1) {
	gettime(&end);
	remain = time - timeval_elapsed(&end, &begin);
	if (remain <= 0)
	    return;
	tv.tv_sec  = remain / 1000;
	tv.tv_usec = (remain % 1000) * 1000;
	if (select (0, NULL, NULL, NULL, &tv) == -1) {
	    if (errno != EINTR 
#if defined(EAGAIN)
		&& errno != EAGAIN
#endif
		) {
		die ("Error in select!\n");
	    }
	}
    }
}
#endif

static TIMESTORAGE epoch_time;

double gettimems(void) {
    TIMESTORAGE a;

    SUB(gettimems, enter);
    gettimeofday(&a, NULL);
    return timeval_elapsed(&a, &epoch_time);
}
void gettime(TIMESTORAGE *a) {
    SUB(gettime, enter);
    gettimeofday(a, NULL);
}
void getepoch(TIMESTORAGE *a) {
    SUB(getepoch, enter);
    if (a != NULL) {
	gettime(a);
	epoch_time = *a;
    }
}

/*  b - a ... <=> operator */
int timeval_cmp(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_cmp, enter);
    if (a->tv_sec > b->tv_sec) {
	return -1;
    } else if (a->tv_sec < b->tv_sec) {
	return 1;
    } else if (a->tv_usec < b->tv_usec) {
	return -1;
    } else if (a->tv_usec > b->tv_usec) {
	return 1;
    }
    return 0;
}

/* add b to a, equivalant to a += b */
int timeval_add(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_add, enter);
    a->tv_sec  += b->tv_sec;
    a->tv_usec += b->tv_usec;
    while (a->tv_usec > 1000000) {
	a->tv_usec -= 1000000;
	a->tv_sec++;
    }
    return 0;
}

/* subtract b from a, equivalant to a -= b */
int timeval_sub(TIMESTORAGE *a, TIMESTORAGE *b) {
    SUB(timeval_sub, enter);
    a->tv_sec  -= b->tv_sec;
    a->tv_usec -= b->tv_usec;
    while (a->tv_usec < 0) {
	a->tv_usec += 1000000;
	a->tv_sec--;
    }
    return 0;
}

/* rc = a - b */
long timeval_elapsed(TIMESTORAGE *a, TIMESTORAGE *b) {
    TIMESTORAGE c;
    SUB(timeval_elapsed, enter);
    c = *a;
    timeval_sub(&c, b);
    return c.tv_sec * 1000 + c.tv_usec / 1000;
}

/* a = 0 */
void timeval_zero(TIMESTORAGE *ptr) {
    SUB(timeval_zero, enter);
    ptr->tv_sec = 0;
    ptr->tv_usec = 0;
}
#endif

/* This should make this process the leader of it's own process group */
void set_process_group(void) {
#if !defined(_WIN32)
    int rc, pgrp, pid;
#if defined(HAVE_SETPGRP)
    pid = getpid();
# if defined(GETPGRP_VOID)
    pgrp = getpgrp();
# else
    pgrp = getpgrp(pid);
# endif    
    if (pid != pgrp) {
# if defined(SETPGRP_VOID)
	rc = setpgrp();
	if (rc < 0) die ("Error in setpgrp: %s(%d)\n", strerror(errno), errno);
# else
	rc = setpgrp(0, pid);
	if (rc < 0) die ("Error in setpgrp: %s(%d)\n", strerror(errno), errno);
# endif
    }
#elif defined(HAVE_SETSID)
    rc = setsid();
    if (rc < 0) die ("Error in setsid: %s(%d)\n", strerror(errno), errno);
#endif
#endif /* _WIN32 */
}

/* Just in case we don't have it (case insensitive string comparison) */
#if !defined(HAVE_STRCASECMP) && !defined(WIN32)
int strcasecmp(const char *a, const char *b) {
    int ua, ub;
    while (*a && *b) {
	ua = toupper(*a);
	ub = toupper(*b);
	if (ua < ub) return -1;
	if (ua > ub) return  1;
	a++; b++;
    }
    if (*a) return -1;
    if (*b) return  1;
    return 0;
}
#endif
