static char *random_c_rcsid = "$Id: random.c,v 1.6 1998/03/11 00:37:20 channui Exp $";

/*
 *	(C)1997 Standard Performance Evaluation Corporation (SPEC)
 *
 *	This suite contains code acquired from several sources who
 *	understand and agree with SPEC's goal of creating fair and
 *	objective benchmarks to measure computer performance.
 *
 *	This copyright notice is placed here only to protect SPEC in the
 *	event the source is misused in any manner that is contrary to
 *	the spirit, the goals and the intent of SPEC.
 *
 *	The source code is provided to the user or company under the
 *	license agreement for the SPEC Benchmark Suite for this suite.
 */

/*****************************************************************
 *                                                               *
 *	Copyright 1991,1992  Legato Systems, Inc.                *
 *	Copyright 1991,1992  Auspex Systems, Inc.                *
 *	Copyright 1991,1992  Data General Corporation            *
 *	Copyright 1991,1992  Digital Equipment Corporation       *
 *	Copyright 1991,1992  Interphase Corporation              *
 *	Copyright 1991,1992  Sun Microsystems, Inc.              *
 *                                                               *
 *****************************************************************/

/*
 * ---------------------- laddis_c_rnd.c ---------------------
 *
 *      Random number generator.
 *
 *.Exported_routines
 *      double  spec_random  (RandomState *state);
 *      long    spec_nrandom (RandomState *state);
 *      void    spec_srandom (RandomState *state, int seed);
 *
 *.Local_routines
 *      None
 *
 *.Revision_History
 *	24-May-97	Chan-Nui    	Re-write to make thread-safe
 *	28-Nov-91	Teelucksingh	ANSI C
 *	01-Aug-91	Wiryaman	laddis_srandom() and laddis_random()
 *					now use spec_srand() and spec_rand()
 *					instead of srandom() and random().
 *      17-Apr-91       Wittle		Created.
 */

/*
 * Here's the source for the random number generator that SPEC uses.
 * The function to be called is "spec_rand" which returns an integer
 * between 1 and MAX_INT-1.
 *
 */

#include <stdlib.h>	/* MC: for malloc() */
#include <stdio.h>
#include <math.h>
#include "random.h"

/*****************************************************************************
 * UNIFORM Distribution                                                      *
 *****************************************************************************/

/*
 * Compute the next random number.
 */
double spec_random(RandomState *state)

/* See "Random Number Generators: Good Ones Are Hard To Find", */
/*     Park & Miller, CACM 31#10 October 1988 pages 1192-1201. */
/***********************************************************/
/* THIS IMPLEMENTATION REQUIRES AT LEAST 32 BIT INTEGERS ! */
/***********************************************************/

#define _A_MULTIPLIER  16807L
#define _M_MODULUS     2147483647L /* (2**31)-1 */
#define _Q_QUOTIENT    127773L     /* 2147483647 / 16807 */
#define _R_REMAINDER   2836L       /* 2147483647 % 16807 */
{
    long	lo;
    long	hi;
    long	test;

    hi = *state / _Q_QUOTIENT;
    lo = *state % _Q_QUOTIENT;
    test = _A_MULTIPLIER * lo - _R_REMAINDER * hi;
    if (test > 0) {
	*state = test;
    } else {
	*state = test + _M_MODULUS;
    }
    return((double) *state / _M_MODULUS);
}

/*
 * Seed the random number generator.
 */
void spec_srandom( RandomState *state, int seed ) {
    *state = seed;
}

/*
 * Returns a random number.
 */
long spec_nrandom( RandomState *state ) {
    spec_random(state);
    return(*state);
}

/*****************************************************************************
 * ZIPF Distribution                                                         *
 *****************************************************************************/

#define	END	0.0

zipf_state_t *spec_zipf_setup(zipf_state_t *state, RandomState *rstate, int size, double Z) {
    int		i;
	double zipf_sum;
    state->size   = size;
    state->rstate = rstate;
 
    state->table=(double *)malloc(sizeof(double)*(size+1));
    if (state->table == NULL) {
	fprintf(stderr, "spec_zipf_setup: can't malloc %d bytes for table\n", 
		sizeof(double)*size);
	exit (1);
    }

	/* compute zipf values for samples 1-n */
    for (i = 1; i <= size; i++) {
	state->table[i] = pow(((double)1.0/((double)i)), Z);
    }

	/* sum the values so we can compute probabilities. */
	/* at the same time, make the values cumulative */
	zipf_sum = 0.0;
	for (i = 1; i <= size; i++) {
		zipf_sum += state->table[i] ;
		state->table[i] = zipf_sum;
    }
	state->table[size] = 0.0;
	state->table[0] = 0.0;

	/* compute probability values by dividing by the sum. */
	/* also reverse the table so we have values starting at 1.0 */
	/*  and descending to 0.0  (this is what spec_zipf needs) */
    for (i = 0; i < size; i++) {
	state->table[i] = 1.0 - (state->table[i]/zipf_sum);
	}

    
    return state;
}

void spec_zipf_free(zipf_state_t *state) {
    if (state->table) {
	free(state->table);
    }
}

int spec_zipf(zipf_state_t *state) {
    double	r;
    int		i;

#ifndef ZIPF_SPIKE
    do{
#endif
        r = spec_random(state->rstate);
        i = 0;
        while (r < state->table[i]) {
	    i++;
		}
#ifndef ZIPF_SPIKE
    } while (i>state->size);
#endif
    return i-1;
}




#ifdef STATIC_NORMAL_DIST
/* Right now, mean and stddev are ignored.  If someone has a good function to
   generate the cdf for normal distributions, let me know... */
/* size=20, mean = 10, stddev = 3.5, will generate numbers from 0 to 20 */
normal_state_t *spec_normal_setup(normal_state_t *state, RandomState *rstate, double mean, double stddev) {

double normal_dist[] = {
    0.002137432,
    0.005064024,
    0.011135458,
    0.022750062,
    0.043238098,
    0.076563771,
    0.126549006,
    0.19568292,
    0.283854542,
    0.387548544,
    0.5,
    0.612451456,
    0.716145458,
    0.80431708,
    0.873450994,
    0.923436229,
    0.956761902,
    0.977249938,
    0.988864542,
    0.994935976,
    1
};
    int i, index = 0;

    state->size   = 1000;
    state->rstate = rstate;

    state->table=(int *)malloc(sizeof(int)*(state->size+1));
    if (state->table == NULL) {
	fprintf(stderr, "spec_normal_setup: can't malloc %d bytes for table\n", 
		sizeof(double)*state->size);
	exit (1);
    }

    for (i = 0; i < state->size; i++) {
	if ((double)i / (double)state->size > normal_dist[index]) {
	    index++;
	}
	state->table[i] = index;
    }

    return state;
}

void spec_normal_free(normal_state_t *state) {
    if (state->table) {
	free(state->table);
    }
}

int spec_nnormal(normal_state_t *state) {
    long rval = spec_nrandom(state->rstate);
    rval = rval % state->size;
    return state->table[rval];
}

#else

/* Guts of this routine are based on: */
/* boxmuller.c           Implements the Polar form of the Box-Muller
                         Transformation

                      (c) Copyright 1994, Everett F. Carter Jr.
                          Permission is granted by the author to use
                          this software for any application provided this
                          copyright notice is preserved.

*/

normal_state_t *spec_normal_setup(normal_state_t *state, RandomState *rstate, double mean, double stddev) {
    state->mean = mean;
    state->stddev = stddev;
    state->rstate = rstate;
    state->use_last = 0;

    return NULL;
}
void spec_normal_free(normal_state_t *state) {
}

double spec_normal(normal_state_t *state) {
        double x1, x2, w, y1;

        if (state->use_last)                   /* use value from previous call */
        {
                y1 = state->y2;
        }
        else
        {
                do {
                        x1 = 2.0 * spec_random(state->rstate) - 1.0;
                        x2 = 2.0 * spec_random(state->rstate) - 1.0;
                        w = x1 * x1 + x2 * x2;
                } while ( w >= 1.0 );

                w = sqrt( (-2.0 * log( w ) ) / w );
                y1 = x1 * w;
                state->y2 = x2 * w;
        }
	state->use_last =! state->use_last;

        return( state->mean + y1 * state->stddev );
}
int spec_nnormal(normal_state_t *state) {
    return (int)(spec_normal(state));
}
#endif
