/*
 * Implementation of ISAPI code for SPECweb99, optimised
 * for Zeus Web Server 3.3.8 and above
 *
 * (c) Zeus Technology Limited 2000-2002.  All rights reserved.
 * 
 * Copyright in the source code ("the Source Code") listed below in whatever
 * form, whether printed electronic compiled or otherwise, belongs to Zeus
 * Technology Limited ("we"/"us").
 * 
 * If you have entered into a written agreement with us under which the Source
 * Code is licensed to you ("Licence"),  you may not use, sell, license,
 * transfer, copy or reproduce the Source Code in whole or in part or in any
 * manner or form other than in accordance with your Licence.  To do so is
 * strictly prohibited and may be unlawful and a serious criminal offence.
 * 
 * If you have not entered into a Licence, disclosure of the Source Code is
 * made "as is".   You may use the Source Code for non-commercial purposes
 * only.  You may distribute the Source Code to individual third parties for
 * their non-commercial purposes only, but only if (1) you acknowledge our web
 * site as the source of the Source Code and include such acknowledgement and
 * our web address (www.zeus.com) in any copy of the Source Code; (2) the
 * Source Code you distribute is complete and not modified in any way and
 * includes this notice; and (3) you inform such third parties that these
 * conditions apply to them and that they must comply with them.
 * 
 * If you have not entered into a Licence, all express and implied warranties,
 * conditions, terms, undertakings and representations, including without
 * limitation as to quality, performance, fitness for purpose, or
 * non-infringement, are excluded to the fullest extent permitted by law.
 * Neither we nor any other person involved in disclosing the Source Code shall
 * have any liabilities whatsoever to you, howsoever arising, in connection
 * with the Source Code, or its use by you.
 * 
 * If you have not entered into a Licence but would like to do so,  please
 * contact us at pepp@zeus.com.
 */

#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <ctype.h>
#include <errno.h>
#include <sys/param.h>  /* for MAXPATHLEN */
#include <sys/types.h>
#include <sys/stat.h>
#include "debug.h"
#include "zeus-common.h"

#define HTML_PRE "\n<pre>\n"
#define HTML_TRAILER "\n</pre>\n</body></html>\n"

/*
 * The following paths are filled in when we load...
 */
const char zeus_home[MAXPATHLEN] =
	"ZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZh"
	"ZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZh";
char global_path_root[MAXPATHLEN] =
	"ZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZz"
	"ZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZz";
char *global_path_tail;
char POST_LOGFILE[MAXPATHLEN];
char UPF_PATH[MAXPATHLEN];
char CAD_PATH[MAXPATHLEN];

typedef struct {
   char *param;
   int checked;
   unsigned int value;
} tuning_t;

/* ----------------------------------------------------------------------
 * Read the requested server variable into a buffer. This routine is
 * only used by parse_request, below. It could be inline, but makes
 * the code look neater as a separate function.
 * -------------------------------------------------------------------- */
static int
read_variable( LPEXTENSION_CONTROL_BLOCK ecb, char *dst, char *env )
{
   int len = 128;
   ecb->GetServerVariable( ecb->ConnID, env, dst, &len );
   return len;
}

/* ----------------------------------------------------------------------
 * The output from every request is preceded by a bit of HTML - 
 * the <html> preamble, and a few server variables: SERVER_SOFTWARE,
 * REMOTE_ADDR, SCRIPT_NAME and QUERY_STRING.
 *
 * This routine generates the above preamble ready for transmission.
 * Note that the memory returned by HSE_REQ_ALLOC_MEM is allocated to
 * this connection only, and will be freed by the server once the
 * connection terminates.
 * -------------------------------------------------------------------- */
static void
create_header_and_trailer( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request)
{
   char *header;
   int len, querylen;
   static int headersize = LARGE_BUFFER;
   static char *part1 = NULL;
   static char *part2 = NULL;
   static char *error_buffer = NULL;
   static int part1len, part2len;

   if ( part1 == NULL ) {
      part1 = malloc( MEDIUM_BUFFER );
      strcpy( part1,
	      "<html>\n"
	      "<head><title>SPECweb99 Dynamic GET & POST Test</title></head>\n"
	      "<body>\n<p>SERVER_SOFTWARE = " );
      read_variable( ecb, part1 + strlen(part1), "SERVER_SOFTWARE" );
      strcpy( part1 + strlen(part1), "\n<p>REMOTE_ADDR = " );
      part1len = strlen( part1 );
      part2 = malloc( MEDIUM_BUFFER );
      strcpy( part2, "\n<p>SCRIPT_NAME = " );
      read_variable( ecb, part2 + strlen(part2), "SCRIPT_NAME" );
      strcpy( part2 + strlen(part2), "\n<p>QUERY_STRING = " );
      part2len = strlen( part2 );
      error_buffer = malloc( LARGE_BUFFER );
   }

   /*
    * Note that there is a serious memory problem if the ALLOC_MEM
    * call fails - by allocating an error_buffer early on in the run,
    * we should survive the failure, although it is likely that the
    * run will fail (as the buffer is now no longer allocated per
    * connection)
    */
   if ( ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_ALLOC_MEM,
                                    &header, &headersize, 0) == FALSE ) {
      debug( "ALLOC_MEM failed! Attempting to continue, but run will probably "
             "fail.\n" );
      header = error_buffer;
   }

   memcpy( header, part1, part1len );
   len = 128;
   len = read_variable( ecb, header + part1len, "REMOTE_ADDR" );
   len += part1len - 1; /* take care of the trailing \0 */
   memcpy( header + len, part2, part2len );
   len += part2len;
   querylen = strlen( ecb->lpszQueryString );
   memcpy( header + len, ecb->lpszQueryString, querylen );
   len += querylen;
   memcpy( header + len, HTML_PRE, sizeof(HTML_PRE) ); /* includes '\0' */

   request->header = header;
   request->headerlen = len + sizeof(HTML_PRE) - 1; /* "\n<pre>\n" */

   request->trailer = HTML_TRAILER;  /* "\n</pre>\n</body></html>\n"; */
   request->trailerlen = sizeof(HTML_TRAILER) - 1;
}

/* ----------------------------------------------------------------------
 * The format of an 'error message page' is the same as all other
 * output: the HTML preamble, the message (as opposed to file) then
 * finally the HTML trailer
 * -------------------------------------------------------------------- */
void
isapi_error( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request,
             const char *fmt, ... )
{
   const char *http_header = "Content-Type: text/html";
   int len;
   char msg[1024];

   va_list args;
   va_start( args, fmt );
   vsprintf( msg, fmt, args );
   va_end( args );

   fprintf( stderr, "SPEC ERROR: %s\n", msg );
   fflush( stderr );

   len = strlen(http_header);
   ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_SEND_RESPONSE_HEADER,
                               0, &len, (DWORD *)http_header);
   len = strlen(request->header);
   ecb->WriteClient( ecb->ConnID, (void *)request->header, &len, 0);
   len = strlen(msg);
   ecb->WriteClient( ecb->ConnID, (void *)msg, &len, 0);
   len = request->trailerlen;
   ecb->WriteClient( ecb->ConnID, (void *)request->trailer, &len, 0);
}

/* Read entire buffer from file descriptor */
int
zeus_readall( int fd, char *buf, int len )
{
   char *p = buf; 
   int s = 0, bytes;
   do {
      bytes = read(fd, p, len - s);
      if (bytes == -1 && errno == EINTR) continue;
      if (bytes == -1) return -1;
      
      s += bytes; p += bytes;
   } while (s < len && bytes > 0);

   return s;
}

/* Write entire buffer to file descriptor */
int
zeus_writeall( int fd, char *buf, int len )
{
   char *p = buf; 
   int s = 0;

   do {
      int bytes = write(fd, p, len - s);
      if (bytes == -1 && errno == EINTR) continue;
      if (bytes == -1) return -1;
      
      s += bytes; p += bytes;
   } while (s < len);

   return 0;
}

/* ----------------------------------------------------------------------
 * The paths are set up by an install script, which runs the binary
 * through sed and replaces the fixed string in path_root with the
 * SPEC docroot.
 * -------------------------------------------------------------------- */
static int
setup_paths( void )
{
   /* Check that spec_install was really run. */
   if ( global_path_root[0] != '/' ) {
      debug( "Bad path '%s' for document root - did you run spec_install?\n",
             global_path_root );
      return FALSE;
   }
   /* set up some default paths... */
   global_path_tail = &global_path_root[ strlen(global_path_root) ];
   sprintf( POST_LOGFILE, "%s/post.log",         global_path_root );
   sprintf( UPF_PATH,     "%s/User.Personality", global_path_root );
   sprintf( CAD_PATH,     "%s/Custom.Ads",       global_path_root );
   return TRUE;
}

#ifdef CHECK_CONFIG
/* ----------------------------------------------------------------------
 * Parse the individual lines of the global.cfg file
 * -------------------------------------------------------------------- */
static tuning_t tuning[] = {
   { "tuning!modules!stats!enabled", 0, 0           }, /* really 'no'  */
   /*{ "tuning!cache_files",         0, 30821       },*/
   { "tuning!cache_max_bytes",       0, 0           },
   { "tuning!cache_small_file",      0, 4096        },
   { "tuning!cache_large_file",      0, 1048576     },
   { "tuning!cache_stat_expire",     0, 31536000    },
   { "tuning!cache_flush_interval",  0, 31536000    },
   { "tuning!cache_cooling_time",    0, 0           },
   { "tuning!sendfile",              0, 1           }, /* really 'yes' */
   /*{ "tuning!sendfile_reservefd",  0, 13000       },*/
   { "tuning!sendfile_minsize",      0, 1           },
   { "tuning!sendfile_maxsize",      0, 2147483647  },
   { "tuning!listen_queue_size",     0, 8192        },
   { "tuning!so_wbuff_size",         0, 1048576     },
   { "tuning!so_rbuff_size",         0, 0           },
   { "tuning!cbuff_size",            0, 65536       },
   { "tuning!softservers",           0, 0           }, /* really 'no'  */
   { "tuning!unique_bind",           0, 1           }, /* really 'yes' */
   { "tuning!bind_any",              0, 0           }, /* really 'no'  */
   /*{ "tuning!num_children",        0, 1           },*/
   { NULL, 0 }
};

static void
parse_cfg( char *line )
{
   int i, value;
   char *ptr;

   for (i=0; tuning[i].param; i++) {
      if ( strncmp(line,tuning[i].param,strlen(tuning[i].param)) != 0 )
         continue;
      ptr = line + strlen(tuning[i].param);
      if ( ! isspace(*ptr) ) continue;
      tuning[i].checked = 1;
      do { ptr++; } while( isspace(*ptr) );
      if ( strncmp(ptr,"yes",3) == 0 ) value = 1;
      else if ( strncmp(ptr,"no",2) == 0 ) value = 0;
      else value = strtoul(ptr,NULL,10);
      if ( value == tuning[i].value ) continue;
      /* error - dump some diagnostics to the web server log via stderr */
      debug( "global.cfg: %s is %d, recommended %d\n",
             tuning[i].param, value, tuning[i].value );
   }
}

/* ----------------------------------------------------------------------
 * Check that the global.cfg file is fairly sensible
 * -------------------------------------------------------------------- */
static void
check_global_cfg( void )
{
   char *zeushome;
   char path[MAXPATHLEN];
   char line[MEDIUM_BUFFER];
   FILE *cfg;
   int i;

   /* these next two checks should never happen */
   if ( (zeushome=getenv("ZEUSHOME")) == NULL ) {
      debug( "check_global_cfg: $ZEUSHOME not set\n" );
      return;
   }
   sprintf( path, "%s/web/global.cfg", zeushome );
   if ( (cfg=fopen(path,"r")) == NULL ) {
      debug( "couldn't open global.cfg for checking\n" );
      return;
   }
   while( fgets( line, MEDIUM_BUFFER - 1, cfg ) != NULL )
      parse_cfg( line );
   fclose( cfg );
   for (i=0; tuning[i].param; i++) {
      if ( tuning[i].checked ) continue;
      debug( "global.cfg: %s is not set!\n", tuning[i].param );
   }
}
#endif /* CHECK_CONFIG */

#ifdef OUT_PROCESS
int
out_process_transmitfile( LPEXTENSION_CONTROL_BLOCK ecb,
                          request_t *request, HSE_TF_INFO *tf )
{
   char *filebuf;
   char http_header[LARGE_BUFFER];
   int len, filelen, fd;
   struct stat st;

   /* read the file */
   if ( (fd = open( path_root, O_RDONLY )) == -1 ) {
      debug( "out_process_transmitfile: couldn't open %s: %s\n",
             path_root, strerror( errno ));
      return FALSE;
   }
   if ( fstat( fd, &st ) == -1 ) {
      debug( "out_process_transmitfile: couldn't fstat fd for %s: %s\n",
             path_root, strerror( errno ));
      return FALSE;
   }
   if ( (filebuf = malloc( st.st_size )) == NULL ) {
      debug( "out_process_transmitfile: couldn't allocate memory!\n" );
      return FALSE;
   }
   filelen = zeus_readall( fd, filebuf, st.st_size );
   if ( filelen < 0 ) {
      debug( "out_process_transmitfile: write to %s failed: %s\n",
             path_root, strerror( errno ));
      return FALSE;
   }
   close( fd );

   /* write out the HTTP headers */
   /* skip past "HTTP 200 OK\n" */
   sprintf( http_header, "%sContent-Length: %d\n",
            &tf->pszStatusCode[12],
            request->headerlen + filelen + request->trailerlen );
   len = strlen( http_header ); /* http headers */
   ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_SEND_RESPONSE_HEADER,
                               "200 OK", &len, (DWORD*)http_header );

   /* write out the SPEC headers */
   len = request->headerlen;
   ecb->WriteClient( ecb->ConnID, (void*)request->header, &len, 0 );

   ecb->WriteClient( ecb->ConnID, (void*)filebuf, &filelen, 0 );

   /* finally write the SPEC trailers */
   len = request->trailerlen;
   ecb->WriteClient( ecb->ConnID, (void*)request->trailer, &len, 0 );

   free( filebuf );

   return TRUE;
}
#endif

/* ----------------------------------------------------------------------
 * Required ISAPI function
 * -------------------------------------------------------------------- */
BOOL WINAPI 
GetExtensionVersion( HSE_VERSION_INFO *pVer )
{
   if ( !setup_paths() )
      return FALSE;
#ifdef CHECK_CONFIG
   check_global_cfg();
#endif
   pVer->dwExtensionVersion = HSE_VERSION;
   strncpy(pVer->lpszExtensionDesc,
           "Zeus optimised SPECweb99 dynamic GET and POST test", 
           HSE_MAX_EXT_DLL_NAME_LEN);
   if ( !zeus_isapi_init() )
      return FALSE;
   return TRUE;
}

/* ----------------------------------------------------------------------
 * Optional ISAPI function
 * -------------------------------------------------------------------- */
BOOL WINAPI
TerminateExtension( DWORD dwFlags )
{
   zeus_isapi_exit();
   return TRUE;
}

/* ----------------------------------------------------------------------
 * Required ISAPI function. This is ISAPI's equivalent to main()
 * -------------------------------------------------------------------- */
DWORD WINAPI
HttpExtensionProc( LPEXTENSION_CONTROL_BLOCK ecb )
{
   request_t request;
#ifdef OUT_PROCESS
   strcpy( request.local_path_root, global_path_root );
   request.local_path_tail = &request.local_path_root[ strlen(request.local_path_root) ];
#endif
   create_header_and_trailer( ecb, &request );
   process_request( ecb, &request );
   /* return and indicate success */
   return HSE_STATUS_SUCCESS;
}
