/*
 * zeus-house.c - implementation of SPECweb99 GET with command
 * (housekeeping tasks) as a CGI extension. Optimised for Zeus Web
 * Server 3.3.8 and above
 *
 * (c) Zeus Technology Limited 2000-2002.  All rights reserved.
 * 
 * Copyright in the source code ("the Source Code") listed below in whatever
 * form, whether printed electronic compiled or otherwise, belongs to Zeus
 * Technology Limited ("we"/"us").
 * 
 * If you have entered into a written agreement with us under which the Source
 * Code is licensed to you ("Licence"),  you may not use, sell, license,
 * transfer, copy or reproduce the Source Code in whole or in part or in any
 * manner or form other than in accordance with your Licence.  To do so is
 * strictly prohibited and may be unlawful and a serious criminal offence.
 * 
 * If you have not entered into a Licence, disclosure of the Source Code is
 * made "as is".   You may use the Source Code for non-commercial purposes
 * only.  You may distribute the Source Code to individual third parties for
 * their non-commercial purposes only, but only if (1) you acknowledge our web
 * site as the source of the Source Code and include such acknowledgement and
 * our web address (www.zeus.com) in any copy of the Source Code; (2) the
 * Source Code you distribute is complete and not modified in any way and
 * includes this notice; and (3) you inform such third parties that these
 * conditions apply to them and that they must comply with them.
 * 
 * If you have not entered into a Licence, all express and implied warranties,
 * conditions, terms, undertakings and representations, including without
 * limitation as to quality, performance, fitness for purpose, or
 * non-infringement, are excluded to the fullest extent permitted by law.
 * Neither we nor any other person involved in disclosing the Source Code shall
 * have any liabilities whatsoever to you, howsoever arising, in connection
 * with the Source Code, or its use by you.
 * 
 * If you have not entered into a Licence but would like to do so,  please
 * contact us at pepp@zeus.com.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/param.h>
#define DEBUG_NAME "HOUSEKEEPING: "
#include "debug.h"

/*
 * The following paths are filled in when we load...
 */
const char zeus_home[MAXPATHLEN] =
	"ZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZh"
	"ZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZhZh";
char path_root[MAXPATHLEN] =
	"ZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZz"
	"ZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZzZz";
static char *path_tail;
static char POST_LOGFILE[MAXPATHLEN];
static char UPF_PATH[MAXPATHLEN];
static char CAD_PATH[MAXPATHLEN];

/*
 * the following are read from the environment
 */
static char *server_software;
static char *remote_addr;
static char *script_name;
static char *query_string;

#define SMALL_BUFFER 32
#define MEDIUM_BUFFER 256
#define LARGE_BUFFER 512

/*
 * The following are filled in before processing the request
 */
static char header[LARGE_BUFFER];
static int headerlen;
static char *trailer = "\n</pre>\n</body></html>\n";
static int trailerlen;

/* ----------------------------------------------------------------------
 * The paths are set up by an install script, which runs the binary
 * through sed and replaces the fixed string in path_root with the
 * SPEC docroot.
 * -------------------------------------------------------------------- */
static void
setup_paths( void )
{
   /* set up some default paths... */
   path_tail = &path_root[ strlen(path_root) ];
   sprintf( POST_LOGFILE, "%s/post.log",         path_root );
   sprintf( UPF_PATH,     "%s/User.Personality", path_root );
   sprintf( CAD_PATH,     "%s/Custom.Ads",       path_root );
}

/* ----------------------------------------------------------------------
 * The output from every request is preceded by a bit of HTML - 
 * the <html> preamble, and a few server variables: SERVER_SOFTWARE,
 * REMOTE_ADDR, SCRIPT_NAME and QUERY_STRING.
 *
 * This routine generates the above preamble ready for transmission.
 * -------------------------------------------------------------------- */
static void
create_header_and_trailer( void )
{
   sprintf( header,
	    "HTTP/1.0 200 OK\nContent-Type: text/html\n\n<html>\n"
	    "<head><title>SPECweb99 Dynamic GET & POST Test</title></head>\n"
	    "<body>\n<p>SERVER_SOFTWARE = %s\n<p>REMOTE_ADDR = %s\n"
	    "<p>SCRIPT_NAME = %s\n<p>QUERY_STRING = %s\n<pre>\n",
	    server_software, remote_addr, script_name, query_string );
   headerlen = strlen( header );
   trailerlen = strlen( trailer );
}

/* ----------------------------------------------------------------------
 * Write a message back to the client, framed with the SPECweb99
 * headers. This is generally only called on failure paths.
 * -------------------------------------------------------------------- */
static void
error( const char *format, ... )
{
   va_list args;
   va_start( args, format );
   printf( header );
   vprintf( format, args );
   printf( trailer );
   va_end( args );
}

/* Write entire buffer to file descriptor */
int
zeus_writeall( int fd, char *buf, int len )
{
   char *p = buf; 
   int s = 0;

   do {
      int bytes = write(fd, p, len - s);
      if (bytes == -1 && errno == EINTR) continue;
      if (bytes == -1) return -1;
      
      s += bytes; p += bytes;
   } while (s < len);

   return 0;
}

/* ----------------------------------------------------------------------
 * The custom ad (CAD) module requires notification after a reset,
 * in order that it can re-read.
 * -------------------------------------------------------------------- */
#ifndef OUT_PROCESS
static void
signal_reset( void )
{
   char path[MAXPATHLEN], buf[SMALL_BUFFER];
   FILE *pidfile;
   int pid;

   /* retrieve the pid from $ZEUSHOME... */
   sprintf( path, "%s/web/internal/pid", zeus_home );
   if ( (pidfile=fopen(path,"r")) == NULL ) {
      debug( DEBUG_NAME "Fatal error - couldn't determine webserver pid: %s\n",
	     strerror( errno ));
      return;
   }
   fgets( buf, SMALL_BUFFER - 1, pidfile );
   pid = atoi( buf );
   fclose( pidfile );

   /* use this to open the /tmp file... */
   sprintf( path, "/tmp/specsignal.%d", pid );
   if ( (pidfile=fopen(path,"r")) == NULL ) {
      debug( DEBUG_NAME
	     "no pid file available yet (CAD module not loaded?)\n" );
      return;
   }
   while( fgets( buf, SMALL_BUFFER - 1, pidfile ) != NULL ) {
      pid = atoi(buf);
      if ( pid > 1 ) {
         debug( DEBUG_NAME "sending sigusr2 to %d...\n", pid );
         kill(pid, SIGUSR2);
      }
   }
}
#endif

/* ----------------------------------------------------------------------
 * Perform a 'reset' command. Note that the Reset command as invoked
 * by the Spec manager program provides five arguments:
 *   maxload, pttime, maxthread, exp, urlroot
 * Note especially that this function requires the arguments to be in
 * this order! (This will be true if called via the manager).
 *
 * A reset command involves re-generating the Custom.Ads file, the
 * User.Personality file and clearing the logfile
 * -------------------------------------------------------------------- */
static void
process_reset( void )
{
   FILE *log;
   char *arg[5], *ptr;
   char path[MAXPATHLEN];
   int i;
   ptr = query_string;
   for (i=0; i<5; i++) {
      if ( (ptr=strchr(ptr,'=')) == NULL ) break;
      arg[i] = ++ptr;
      if ( i < 4 ) {
         if ( (ptr=strchr(ptr,'&')) == NULL ) break;
         *(ptr++) = '\0';
      }
   }
   if ( i != 5 ) {
      error( "Reset: invalid number of arguments" );
      return;
   }
   /* strip comma out of exp= string (arg[3]) */
   if ( (ptr=strchr(arg[3],',')) == NULL ) {
      error( "Reset: bad exp= argument" );
      return;
   }
   *ptr = ' ';
   *path_tail = '\0';
   sprintf(path, "%s/upfgen99 -C %s -n %s -t %s", path_root, path_root,
           arg[0], arg[2]); /* maxload, maxthread */
   system(path);
   sprintf(path, "%s/cadgen99 -C %s -e %s -t %s %s", path_root, path_root,
           arg[1], arg[2], arg[3]); /* pttime, maxthread, exp */
   system(path);
   /* truncate the POST logfile */
   if ( (log = fopen( POST_LOGFILE, "w" )) == NULL ) {
      error( "Reset: couldn't create POST logfile: %s", strerror( errno ));
      return;
   }
   fprintf(log, "%10d\n", 0);
   fclose(log);
   /* now attempt to signal to the CAD ISAPI module... */
#ifdef OUT_PROCESS
   /* granularity of mtime is 1 second; ensure we re-read Custom.Ads etc */
   sleep( 1 );
#else
   signal_reset();
#endif
   /* actually the following is really just a message, and not an error */
   error( "Reset: completed successfully" );
}

/* ----------------------------------------------------------------------
 * Perform a 'fetch' command - send the post.log file back to the
 * client, with appropriate HTML headers and trailers.
 * Note that we don't use the TRANSMIT_FILE extension to do this,
 * as the post.log file will change during the run, which will tend
 * to defeat the web server caching. At any rate, this function is
 * not performance critical, as it is only ever executed at the end
 * of a SPEC run.
 * -------------------------------------------------------------------- */
static void
process_fetch( void )
{
   int fd, len;
   char *buf;

   if ( (buf = malloc( 1048576 )) == NULL ) {
      error( "Fetch: couldn't allocate memory!" );
      return;
   }
   if ( (fd = open( POST_LOGFILE, O_RDONLY )) == -1 ) {
      error( "Fetch: couldn't open post.log file: %s",
             strerror( errno ));
      return;
   }

   /* write out the HTTP + SPEC headers */
   zeus_writeall( 1, header, headerlen );

   /* now write out the file data in one-megabyte chunks */
   while( (len=read(fd,buf,1048576)) )
      zeus_writeall( 1, buf, len );

   /* finally write the SPEC trailers */
   zeus_writeall( 1, trailer, trailerlen );

   close(fd);
}

/* ----------------------------------------------------------------------
 * The query string started with command/ so see which 'housekeeping'
 * task to perform. There are only two possibilities: reset (used to
 * generate various files at the start of the test) and fetch (used to
 * fetch the post log at the end of the test).
 * -------------------------------------------------------------------- */
int
main( void )
{
   setup_paths();
   server_software = getenv( "SERVER_SOFTWARE" );
   remote_addr     = getenv( "REMOTE_ADDR" );
   script_name     = getenv( "SCRIPT_NAME" );
   query_string    = getenv( "QUERY_STRING" );
   create_header_and_trailer();

   if ( strncmp(query_string+8,"Reset",5) == 0 ) {
      process_reset();
      return 0;
   }
   if ( strncmp(query_string+8,"Fetch",5) == 0 ) {
      process_fetch();
      return 1;
   }
   error( "invalid housekeeping request: %s", query_string );
   return 1;
}
