/* (C) 1997 Standard Performance Evaluation Corporation */
#include "client.h"

/*------------------------------------------------------*/
/*	             Revisions				*/
/*------------------------------------------------------*/
/* 08/23/2001 - Paula Smith	Compaq			*/
/*                                                      */
/* Added Support for SSL/HTTPS				*/
/*------------------------------------------------------*/
/* 04/28/2001 - gcocco software, inc.  Gerry Cocco      */
/*                                                      */
/* change made to allow use of logical interfaces       */
/* - added "-g ip_addr" parameter                       */
/* - added "-C ip_addr" parameter for control port      */
/* - changed "bind" to use an actual ip_address         */
/*           instead of IN_ADDR_ANY                     */
/*------------------------------------------------------*/

/* Change Added By Pallab */
#ifdef HAVE_RLIMIT
#include <sys/resource.h>
#endif
  
static char *client_c_rcsid = "$Id: client.c,v 1.15 1998/03/05 05:16:32 channui Exp $";

/* This needs to handle overlapping copies */
#define safe_memcpy memcpy

/* Pointers for Thread Specific Data */

TSP OUT_buf  =NULL;  /* Output buffer to LOG to if LOG_OUT is set,
                          this just makes LOG files and control output
			  consistant */
TSP LOG_file =NULL;  /* File to log to for this thread */

TSP Host_lock=NULL;  /* Mutex to lock for non-thread-safe functions,
                          specifically gethostbyname */

void *SHM = NULL;

/*char *ProgramName = "SPEC-WEB-99-v1.02";*/
/*char *ProgramName = "SPECweb99_SSL-v1.0";*/
char *ProgramName = "SPECweb99_SSL-v1.0.alt.WIN32";

int Setup_STDIN_cs(cl_options_t *options);

int control_socket_count = 0; /* count of total number of control sockets 
                                 ever opened */
int die_now = 0;  /* if > 0 it requests that we die */
int dying   = 0;  /* if > 0 states that we are now attempting to die */

cl_options_t options;

#ifdef HTTPS_SUPPORT
SSL_CTX *c_ctx = NULL;		/* Client's ssl context */
SSL_METHOD *ssl_method;

BIO *bio_err;
BIO *bio_stdout;
#endif

RandomState ranseed; /* seed for global random numbers */


/* Prints out a short list of options */
void usage(char *name) {
    SUB(usage, enter);
#ifdef WIN32
    /*
     * MC: if we are a Windows NT service then we don't have a console 
     *     so don't do the printf().
     */
    if (MC_Main_Options->daemonize)
            MC_AddToMessageLog(EVENTLOG_ERROR_TYPE, "Invalid command line parameters.\n");
    else
    {
#endif /* WIN32 */
    printf("Usage: %s [options]\n", name);
    printf("       -h    #   this help screen\n");
    printf("       -D    #   debugging level\n");
    printf("       -g    #   logical IP address to listen to (test traffic)\n");
    printf("       -C    #   logical IP address to listen to (control port)\n");
    printf("       -p    #   control port to listen on\n");
    printf("       -i        use to indicate client is being run out of inetd\n");
    printf("       -t    #   daemon idle timeout (set larger than anticipated run)\n");
    printf("       -c        start a control session on stdin\n");
#ifdef WIN32
    printf("       -z install|start|stop|remove actions for the SPECweb client service\n");
    printf("       -Z    #   run the client but not as a Win32 system service\n");
#endif /* WIN32 */
    printf("       -d        start in background as a daemon\n");
#ifdef USE_SYSV_IPC
    printf("       -m    #   size of shared memory segment\n");
#endif      
    printf("       -w   dir  work directory\n");
    printf("       -s    #   Deck size\n");
    printf("       -S    #   Deck cache size\n");
#ifdef DEBUG
    /* force core dump */
    abort();
#else
    exit(1);
#endif
#ifdef WIN32 /*MC*/
    }
#endif /* WIN32 */
}

/* Parse the command line arguments and sets default values */
int ParseArguments(int argc, char *argv[], cl_options_t *options) {
    extern char *optarg;
    int c;

    SUB(ParseArguments, enter);
    memset(options, 0, sizeof(*options));
    options->shmsize=DEFAULT_SHMSIZE;
    options->Control_port=-1;
    options->idle_timeout=-1;
    options->Control_fd  =-1;
    options->deck_size   = 100;
    options->cache_size  = 10;

    strcpy(options->IPAddress, "INADDR_ANY"); /* Assume: unitialized gmc */
    strcpy(options->CIPAddress,"INADDR_ANY"); /* gmc */

#ifdef WIN32
    options->daemonize = 1;	/* default for Windows NT is to run as system service */
    options->mc_svc_action = MC_SVC_NONE;
#endif /* WIN32 */

    while ((c = getopt(argc, argv, 
#ifdef USE_SYSV_IPC
	    "m:"
#endif
#ifdef WIN32
	    "z:Z"
#endif
            "cdw:it:p:D:hs:S:g:C:")) != EOF) {
	switch (c) {
	    case 'D': Debug_level = options->Debug_level = atoi(optarg);      break;
	    case 'p': options->Control_port    = atoi(optarg);      break;
	    case 'i': options->inetd_child     = 1;                 break;
	    case 't': options->idle_timeout    = atoi(optarg)*1000; break;
	    case 'c': options->console_session = 1;                 break;
	    case 'd': options->daemonize       = 1;                 break;
#ifdef WIN32
	    case 'Z': options->daemonize       = 0;                 break;
	    case 'z': 
			if (_stricmp(optarg, "install") == 0)
				options->mc_svc_action = MC_SVC_INSTALL;
			else if(_stricmp(optarg, "start") == 0)
				options->mc_svc_action = MC_SVC_START;
			else if(_stricmp(optarg, "installstart") == 0)
				options->mc_svc_action = MC_SVC_INSTALL | MC_SVC_START;
			else if(_stricmp(optarg, "stop") == 0)
				options->mc_svc_action = MC_SVC_STOP;
			else if(_stricmp(optarg, "remove") == 0)
				options->mc_svc_action = MC_SVC_REMOVE;
			else
                        {
				usage(argv[0]);
                                return 1;
                        }
                        options->daemonize = 0;
			break;
#endif /* WIN32 */
	    case 'w': options->workdir         = optarg;            break;
	    case 's': options->deck_size       = atoi(optarg);      break;
	    case 'S': options->cache_size      = atoi(optarg);      break;
	    case 'm': options->shmsize         = atoi(optarg);      
		      {
			    int type = strlen(optarg)-1;
			    if (tolower(optarg[type])=='k') {
				options->shmsize *= 1024;
			    } else if (tolower(optarg[type])=='m') {
				options->shmsize *= 1024 * 1024;
			    }
		      }
		      break;
            case 'g': /* gmc: add this for -g processing */
                      if (strlen(optarg) > IPADDR_SIZE) {
                         printf("Error: ipaddress parameter bad: %s \n",
                                 optarg);
                         usage(argv[0]);
                         return 1;
                      }
                      strcpy(options->IPAddress,optarg);
                      LOG(LOG_INFO, "INFO: using bind_addr    \'%s\'\n",
                              options->IPAddress);
                      break;
            case 'C': /* gmc: add this for -C processing */
                      if (strlen(optarg) > IPADDR_SIZE) {
                         printf("Error: ipaddress parameter bad: %s \n",
                                 optarg);
                         usage(argv[0]);
                         return 1;
                      }
                      strcpy(options->CIPAddress,optarg);
                      LOG(LOG_INFO, "INFO: using control_addr \'%s\'\n",
                              options->CIPAddress);
                      break;
	    case 'h':
	    case '?':
	    default:
		usage(argv[0]);
                return 1;
	}
    }

    /* If no control port is specified, look up the service, for now keep
       the port number in host order */
    if (options->Control_port <= 0) {
	struct servent *sp;

	sp = getservbyname(SERVICE_NAME, "tcp");
	if (sp != NULL) {
	    options->Control_port = ntohs(sp->s_port);
	} else {
	    options->Control_port = DEFAULT_PORT;
	}
    }

    /* If no working directory is specified, default to the current one */
    if (options->workdir == NULL) {
	char buf[PATH_MAX], buf2[PATH_MAX];
	if (getcwd(buf, PATH_MAX) == NULL) {
#ifdef WIN32
            if (MC_Main_Options->daemonize)
                return 1; /* MC: let MC_ServiceMain() exit gracefully */
#endif /* WIN32 */
            die ("Error in getcwd: %s(%d)\n", strerror(errno), errno);
	}
#ifdef WIN32
	sprintf(buf2, "%s\\%s", buf, WORKDIR_NAME);     /* MC */
#else
        sprintf(buf2, "%s/%s", buf, WORKDIR_NAME);
#endif /* WIN32 */
	options->workdir = strdup(buf2);
    }

    /* If we are spawned from inetd, idle for 5 minutes with no connections */
    if (options->inetd_child && options->idle_timeout < 0) {
	options->idle_timeout=300;
    }
    return 0;
}

/* Create opinters to our thread specific data */
int InitializeThreadSystem(cl_options_t *options) {
    int rc = 0;

    SUB(enitializeThreadSystem, enter);


    /* pthread_init() should have been called by InitChildLayer() */
    OUT_buf  = TSPCreate();
    LOG_file = TSPCreate();
    Host_lock = MutexCreate();
    SHM       = SHMinit(options->shmsize);
    return rc;
}


/* This is our generic signal handler, it handles all of our signals */
RETSIGTYPE signal_handler(int sig) {
/*    SUB(signal_handler, enter); */
      int rc;
#if !defined(USE_POSIX_THREADS) && !defined(HAVE_SIGACTION) && !defined(_WIN32)
    signal(sig, signal_handler);  /* Unreliable signals need to reset handler */
#endif

    switch (sig) {
#if defined(USE_POSIX_THREADS)
    case SIGTERM:
    	/* SIGTERM is sent to kill off wayward children or by the user */
	if (!pthread_equal(main_thread, pthread_self())) {
		LOG(LOG_ERROR|LOG_STDERR, "Received SIGTERM and exiting\n");
		exit(1);
	}
    case SIGINT:
    	/* SIGINT is an interrupt by the user*/
	if (die_now == 0) {
	  LOG(LOG_ERROR|LOG_STDERR, 
            "Received SIGINT, attempting to shut down!\n");
	    die_now = 1;
	}
	rc = pthread_kill(main_thread, SIGALRM);
	if (rc)
	  LOG(LOG_ERROR|LOG_STDERR,"pthread_kill failed: %s.\n", strerror(rc));
	break;
    case SIGPIPE:
	if (!pthread_equal(main_thread, pthread_self())) {
	    LOG(LOG_ERROR|LOG_STDERR, "Received SIGPIPE and exiting\n");
#	ifdef DEBUG
            abort(); /* force core dump */
#	else
            exit(1);
#	endif
	}
    case SIGCHLD: 
	{
	    int status;
	    wait(&status);
	    break;
	}
    case SIGALRM:
        /* The alarm signal is used to keep from waiting forever when we */
        /* are shutting down */
	break; 


#elif defined(USE_SYSV_IPC)
    case SIGTERM:
	if (main_thread!=getpid()) {
		LOG(LOG_ERROR|LOG_STDERR, "Received SIGTERM and exiting\n");
		exit(1);
	}
    case SIGINT:
	if (die_now == 0) {
	  LOG(LOG_ERROR|LOG_STDERR, "Received SIGINT, attempting to shut down!\n");
	  die_now = 1;
	}
	kill(main_thread, SIGALRM);
	break;
    case SIGPIPE:
	if (main_thread!=getpid()) {
	    LOG(LOG_ERROR|LOG_STDERR, "Received SIGPIPE and exiting\n");
#	ifdef DEBUG
            abort(); /* force core dump */
#	else
            exit(1);
#	endif
	}

    case SIGCHLD: 
    {
	int status;
	wait(&status);
	break;
    }

    case SIGALRM:
        /* The alarm signal is used to keep from waiting forever when we */
        /* are shutting down */
	break; 

#elif defined(_WIN32)
    case SIGTERM:
	LOG(LOG_ERROR|LOG_STDERR, "Received SIGTERM and exiting\n");
	exit(1);
		
    case SIGINT:
	if (die_now == 0) {
	  LOG(LOG_ERROR|LOG_STDERR, "Received SIGINT, attempting to shut down!\n");
	  die_now = 1;
	}
	break;

#endif
    default:
	die ("Unknown signal (%d)\n", sig);
    }
}



/* If we are using POSIX threads use a thread to handle all signals.  This
   allows us to handle signals synchronously, which is much nicer. */
#if defined(USE_POSIX_THREADS)
void *signal_thread(void *ptr) {
    sigset_t set;
    int sig;
    int rc;

    SUB(signal_thread, enter);
    sigemptyset(&set);
    sigaddset(&set, SIGINT);
    sigaddset(&set, SIGCHLD);
    sigaddset(&set, SIGTERM);
    sigaddset(&set, SIGPIPE);
/*    sigaddset(&set, SIGALRM); */
#ifdef HAVE_SIGTHREADMASK
    sigthreadmask(SIG_BLOCK, &set, NULL);
#else
    rc = pthread_sigmask(SIG_BLOCK, &set, NULL); 
    if (rc)
        LOG(LOG_ERROR|LOG_STDERR,"pthread_sigmask failed: %s.\n", strerror(rc));
#endif

    while (1) {
	LOG(LOG_TRACE, "Waiting for signal\n");
        sigwait(&set, &sig);
	LOG(LOG_TRACE, "Handling signal %d\n", sig);
        signal_handler(sig);
    }
    return ((void *) NULL);
}
#endif

int InitializeSignals(cl_options_t *options) {
    int rc = 0;
#if defined(USE_POSIX_THREADS)
    pthread_t th;
#endif
#if defined(HAVE_SIGACTION)
    struct sigaction sa;
#endif

    SUB(InitializeSignals, enter);

#if defined(USE_POSIX_THREADS)
    if ((rc = pthread_create(&th, NULL, signal_thread, NULL) != 0)) {
	die ("Can't create signal handling thread!: %s\n", strerror(rc));
    }
# if defined(HAVE_SIGACTION)
    sa.sa_handler = signal_handler;
    sa.sa_flags   = 0;
    sigemptyset(&sa.sa_mask);
    sigaction(SIGALRM, &sa, NULL);
# else
    signal(SIGALRM, signal_handler);
# endif

#elif defined(_WIN32)
    /* These are probably not correct, I haven't tried compiling under NT yet */
    signal(SIGINT, signal_handler);
    signal(SIGTERM, signal_handler);
#elif defined(HAVE_SIGACTION)
    sa.sa_handler = signal_handler;
    sa.sa_flags   = 0;
    sigemptyset(&sa.sa_mask);

    rc  = sigaction(SIGINT,  &sa, NULL) ||
          sigaction(SIGTERM, &sa, NULL) ||
          sigaction(SIGUSR1, &sa, NULL) ||
          sigaction(SIGALRM, &sa, NULL) ||
          sigaction(SIGPIPE, &sa, NULL) ||
          sigaction(SIGCHLD, &sa, NULL);

    if (options->daemonize) {
	sa.sa_handler = SIG_IGN;
	sa.sa_flags   = 0;
	sigemptyset(&sa.sa_mask);

	rc |= sigaction(SIGTTOU, &sa, NULL) ||
	      sigaction(SIGTTIN, &sa, NULL) ||
	      sigaction(SIGTSTP, &sa, NULL);
    }

#else
    rc  = signal(SIGINT,  signal_handler) ||
          signal(SIGTERM, signal_handler) ||
          signal(SIGUSR1, signal_handler) ||
          signal(SIGALRM, signal_handler) ||
          signal(SIGPIPE, signal_handler) ||
          signal(SIGCHLD, signal_handler);

    if (options->daemonize) {
	rc |= signal(SIGTTOU, SIG_IGN) ||
	      signal(SIGTTIN, SIG_IGN) ||
	      signal(SIGTSTP, SIG_IGN);
    }
#endif

    return 0;
}


#if !defined(_WIN32)
int InitializeDaemon(cl_options_t *options) {
    int i, pid, fd, rc;

    if (getppid() == 1)
	return 0;

    /* Set umask so everyone can read what we do, but no one can change it */
    umask(0022);

    /* Background ourselves */
    if ( (pid = fork()) < 0 ) {
	die ("Can't fork: %s(%d)\n", strerror(errno), errno);
    } else if (pid > 0) {
	exit(0); /* parent */
    }

    /* Make ourselves a new process group, this should override whatever
       was done in InitChildLayer */
    set_process_group();

    /* Try to dump our controlling tty, this prevents signals from the tty,
     * like suspend, from reaching us */
#ifdef TIOCNOTTY
    if ((fd = open("/dev/tty", O_RDWR)) >= 0) {
	ioctl(fd, TIOCNOTTY, NULL);
	close(fd);
    }
#else
    /* This time for sure! */
    if ( (pid = fork()) < 0 ) {
	die ("Can't fork: %s(%d)\n", strerror(errno), errno);
    } else if (pid > 0) {
	exit(0); /* parent */
    }
#endif

    /* Find out what our log file descriptor is */
    fd = -1;
    if (LOG_file != NULL) {
	FILE *fp = (FILE *)TSPGet(LOG_file);
	if (fp != NULL)
	    fd = fileno(fp);
    }
/*#ifdef FCLOSE_STDOUT*/
    fclose (stdout);
/*#endif*/

    /* Close all file descriptors except for our control socket and our
       log file descriptor */
    for (i = 0; i < NOFILE; i++) {
	if (i != options->Control_fd && i != fd)
	    close(i);
    }
    return 0;
}
#endif

/* Initialize Working Directory and move to it */
int InitializeWorkDir(cl_options_t *options) {
    int rc;
    FILE *fp;

    SUB(InitializeWorkDir, enter);

    /* Make sure directory exists */
    rc = mkdir(options->workdir, 0755);
    if (rc != 0 && errno != EEXIST && errno != 0) {
	die ("Can't Initialize Working Directory '%s': %s(%d)\n", 
	     options->workdir, strerror(errno), errno);
    }

    /* Change into working directory */
    if (chdir(options->workdir)) {
	die ("Can't chdir to Working Directory '%s': %s(%d)\n", 
	     options->workdir, strerror(errno), errno);
    }
    /* Close the log file if it is already open */
    if (LOG_file != NULL) {
	fp = (FILE *)TSPGet(LOG_file);
	if (fp != NULL)
	    fclose(fp);
    }

    /* Open a new log file */
    fp = fopen (LOGFILE_NAME, "w+");
    if (fp == NULL) {
	die("Can't open log file '%s'!: %s(%d)\n", LOGFILE_NAME, strerror(errno), errno);
    }
    TSPSet(LOG_file, fp);
#ifdef SPEC_DEBUG
    setbuf(fp, NULL); /* Unbuffer the log file */
#endif

    return 0;
}

/* Create/Initialize the Port that we accept connections on */
int InitializeControlPort(cl_options_t *options) {
    struct protoent FAR *proto;
    struct sockaddr_in accept_addr;
    int    serv_port, servfd;
    int    no, tcp_proto;

    SUB(InitializeControlPort, enter);

    /* If we are spawned from inetd with the "wait" option then everything
       should be OK already */
    if (options->inetd_child) {
	options->Control_fd = 0;
	return 0;
    }
    serv_port = options->Control_port;

    /* Get the protocol number */
    LOG(LOG_SOCKET, "Creating Control Socket\n");
    if ((proto = getprotobyname ("tcp")) == NULL) {
	warn("Unknown protocol 'tcp', trying 6\n");
	tcp_proto = 6;
    } else {
	tcp_proto = proto->p_proto;
    }

    /* Create the socket */
    if (( servfd = socket(AF_INET, SOCK_STREAM, tcp_proto)) < 0)
	die ("Can't create socket\n");

    /* If this fails that's ok.  We only use this to try to re-bind quickly */
    no = 1;
    setsockopt(servfd, SOL_SOCKET, SO_REUSEADDR, (const char *)&no, sizeof(no));

    /* Set up the address */
    memset(&accept_addr, 0, sizeof(accept_addr));
    accept_addr.sin_family      = AF_INET;

    /*gmc: here we need to substitute for INADDR_ANY if -C is specified */
    if (strcmp(options->CIPAddress,"INADDR_ANY") != 0) {
       /*gmc: new way - listen only for a specific ipaddress */
       accept_addr.sin_addr.s_addr = inet_addr(options->CIPAddress);
    } else {
       /*gmc: old way - accept from any ip address */
       accept_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    }
    accept_addr.sin_port        = htons((short)serv_port);


    /* Bind ourselves to the port and be willing to accept as many people
       as we can on the queue */
    if (bind(servfd, (struct sockaddr *)&accept_addr, sizeof(accept_addr)) < 0)
	die ("Can't bind to port %d\n", serv_port);
    if (listen(servfd, SOMAXCONN) < 0) /* Maximum amount */
	die ("Can't listen to port %d\n", serv_port);

    options->Control_fd = servfd;
    
    return 0;
}

/* This function is called whenever you want to cleanly close down a control
   socket */
int CloseControlSocket( control_socket_t *cs ) {
    SUB(CloseControlSocket, enter);
    
    /* Mark the socket as pending close */
    cs->want_close = 1;

    /* Iff the socket has had the init command issued, then mark the state
       as abort.  This will cause any children to exit after their current
       transaction */
    if (cs->lock != NULL) {
	MutexLock(cs->lock);
	cs->state.abort = 1;
	MutexUnlock(cs->lock);
    }

#ifdef WIN32
    /*
     * Interrupt any worker threads that might be blocked on a socket 
     * read or write call.
     */
    SetEvent(cs->mc_stop_event);
#endif /* WIN32 */

    return 0;
}


/* ControlAction is the standard function called when the control socket
   has been selected on.  The do(read/write/except) arguments just indicate
   the return from select, docommand is set if you want to force the
   processing of a command. */

int ControlAction (int doread, int dowrite, int doexcept, int docommand,
		   control_socket_t *cs) {
    int rc;
    char *ptr, *tmp;

    SUB(ControlAction, enter);

    /* If there is an exception thrown then we can't handle it, just close
       the socket and hope for the best */
    if (doexcept) {
	LOG(LOG_ERROR, "Control Socket %d: exception\n", cs->num);
	CloseControlSocket(cs);
	return 1;
    }

    if (dowrite) {
	/* We use send because NT can't handle read/write on sockets */
	if (cs->console_session) {
	    rc = write(cs->fd, cs->output.buffer+cs->output.start, 
	               cs->output.len);
	} else {
	    rc = send(cs->fd, cs->output.buffer+cs->output.start, 
	              cs->output.len, 0);
	}
#ifdef WIN32
        if (rc <= 0)
        {
            LOG(LOG_STDERR, "ControlAction(): remote close on control socket %d\n", cs->num);
            SetEvent(cs->mc_stop_event);
#else
	if (rc < 0) {
	    /* We had an error */
	    if (errno != EINTR 
#if defined(EAGAIN)
		&& errno != EAGAIN
#endif /* EAGAIN */
		) {
		die ("Error writing to fd %d: %s(%d)", cs->fd, strerror(errno), errno);
	    }
	} else if (rc == 0) {
	    /* We couldn't write any data, the socket's probably closed,
	       shut it down now */
#endif /* WIN32 */
	    CloseControlSocket(cs);
	    return 1;
	} else {
	    /* Update the pointers in the buffer */
	    cs->output.len   -= rc;
	    cs->output.start += rc;
	    if (cs->output.len == 0) {
		cs->output.start = 0;
	    }
	}
	if (cs->output.len == 0 && cs->must_exit == 1) {
	    /* If this socket is tagged to be closed and we've written out
	       all of the data, then close it now */
	    CloseControlSocket(cs);
	    return 1;
	}
    }

    if (doread) {
	/* We use recv because NT can't handle read/write on sockets */
	if (cs->console_session) {
	    rc = read(cs->fd, cs->input.buffer+cs->input.start+cs->input.len, 
		    sizeof(cs->input.buffer)-cs->input.start-cs->input.len-1);
	} else {
	    rc = recv(cs->fd, cs->input.buffer+cs->input.start+cs->input.len, 
		    sizeof(cs->input.buffer)-cs->input.start-cs->input.len-1,0);
	}

#ifdef WIN32 /*MC*/
        if (rc <=0)
        {
            LOG(LOG_STDERR, "ControlAction(): remote close on control socket %d\n", cs->num);
            SetEvent(cs->mc_stop_event);
#else
	if (rc == 0) {
	    /* 0 length read, the socket is probably closed */
#endif /* WIN32 */
	    CloseControlSocket(cs);
	    return 1;
	}
	/* NT telnet doesn't echo locally?  This will echo back anything we
	   send back to the telnet client.  This is for debugging, echoing
	   the commands back will mess up manager */
	/* MC: NT telnet will echo locally, go to the terminal->preferences menu
	 *     and turn on "local echo"
	 */
#if defined(NEED_SOCKET_ECHO)
	buf_write(&cs->output, cs->input.buffer+cs->input.start+cs->input.len, rc);
#endif
	cs->input.len += rc;
	cs->input.buffer[cs->input.start+cs->input.len] = 0;
    }

    /* If we read anything, or we were ordered to, process a command from
       the input buffer */
    if (doread || docommand) {
	/* Mark the current output buffer, as this control sockets */
	TSPSet(OUT_buf, &cs->output);

	/* If we aren't accepting input then abort, otherwise check to
	   see if there is a newline in the input buffer */
	while (cs->input_mode != CS_INPUT_NONE &&
	       (tmp = strchr(cs->input.buffer+cs->input.start, '\n'))) {
	    int len;
	    int num_words;
	    word_list *words;

	    /* Mark the beginning and length of the command we found */
	    ptr = cs->input.buffer + cs->input.start;
	    len = tmp-(ptr) + 1;

	    /* Depending on our input mode ... */
	    switch (cs->input_mode) {
	    case CS_INPUT_FILE: 
		/* Files are just written out until we find a line containing
		   just a period */
		if (*ptr == '.' && (*(ptr+1) == '\n' || *(ptr+1) == '\r')) {
		    buf_printf(&cs->output, "200 OK\n");
		    cs->input_mode = CS_INPUT_COMMAND;
		    cs->need_command = 1;
		    close(cs->input_file);
		} else {
		    write(cs->input_file, ptr, len);
		}
		cs->input.len   -= len;
		cs->input.start += len;
		break;
	    case CS_INPUT_COMMAND:
		/* Commands are broken up into words and then passed
		   to the Request Handlers */
		num_words = break_on_whitespace(ptr, &words);
		if (num_words) { /* If there were any words on that line */
		    ParseRequest(words, cs);
		    free_whitespace_buf(words);
		}
		cs->input.len   -= len;
		cs->input.start += len;
		break;
	    default:
		die ("Illegal input mode! %d\n", cs->input_mode);
	    }
	}
	/* If we exhaust the buffer, then reset the pointers to the
	   beginning */
	if (cs->input.len <= 0) {
	    *(cs->input.buffer) = cs->input.len = cs->input.start = 0;
	}
	/* We've processed a command, so we don't *have* to again */
	cs->need_command = 0;  

	/* If the buffer is full, try to move everything back to the
	    beginning, to free up space at the end, otherwise the
	    buffer has overflowed and we need to abort now. */
	if (cs->input.start + cs->input.len >= sizeof(cs->input.buffer) - 1) {
	    if (cs->input.start > 0) {
		safe_memcpy(cs->input.buffer, 
			    cs->input.buffer+cs->input.start, cs->input.len+1);
		cs->input.start = 0;
	    } else {
		buf_printf(&cs->output, "403 ERROR: BUFFER OVERFLOW\n");
		cs->must_exit = 1;
	    }
	}
	TSPSet(OUT_buf, NULL); /* Reset the logging buffer */
    }
    return 0;
}

/* HandleControlActivity is the big event loop that processes all of the
   commands, if we ever go to a thread-only release, this should be replaced
   with something that spawns off a thread to handle each connection */

int HandleControlActivity(cl_options_t *options) {
    TIMESTORAGE current_time, lastaction_time;
    fd_set rin, rout, win, wout, ein, eout;
    SOCKET servfd, maxfd;
    int rc;
    lg_data_t		*lg_ptr, *lg_next;

#ifdef WIN32 /*MC*/
    BOOL                mc_want_alarm_event;
    char                mcbuf[512];
    LARGE_INTEGER       mc_alarm_time;
    int                 mc_select_want;
    DWORD               mcret;
#endif

    SUB(HandleControlActivity, enter);

    /* We've only just begin, so mark this time as an "action" */
    gettime(&lastaction_time); 

    servfd = options->Control_fd;
    
    /* Do this until someone sends as a command over a control socket */
    while (1) {
	int newsock_len;
	struct sockaddr_in newsock_addr;
	control_socket_t *ptr, *next, *last;
	struct timeval timeval_tmp, *timeval_ptr;
	TIMESTORAGE *epoch_ptr;
	unsigned long global_next_time;
	int want_close;

	/* We figure out what connections we want to monitor from scratch
	   each time */
	FD_ZERO(&rin);
	FD_ZERO(&win);
	FD_ZERO(&ein);
	maxfd = 0;

#ifdef WIN32
        MC_NControlEvents = 1;  /* MC: the ProgramShutdownEvent event is already set */
#endif /* WIN32 */

	timeval_ptr = NULL;
	/* DEC Needs this to be INT_MAX rather than LONG_MAX*/
	global_next_time = INT_MAX;
	epoch_ptr = &options->epoch_time;

	gettime(&current_time); /* Get the current time for this loop */

	/* If we are marked that we should die, then mark all of the
	 * control sockets to be closed.  If we are already dying then do
	 * nothing.  If neither of those cases are true, put the server fd in
	 * the select mask, so that we can accept new connections */
	if (dying == 1) {
	    LOG(LOG_TRACE, "Still dying\n");
	} else if (die_now > 0)  {
	    LOG(LOG_TRACE, "Recieved a request to terminate the client\n");
	    for (ptr = options->Control_socks; ptr; ptr = ptr->next) {
		CloseControlSocket(ptr);
	    }
	    dying = 1;
	} else {
	    LOG(LOG_TRACE, "Will accept control sockets\n");
#ifdef WIN32 /*MC*/
            if (WSAEventSelect(servfd, options->mc_select_event, FD_READ|FD_ACCEPT) == SOCKET_ERROR)
                die("WSAEventSelect() failed, errno = %d\n", WSAGetLastError());
            MC_Control_Events[MC_NControlEvents++] = options->mc_select_event;
#endif /* WIN32 */
            FD_SET(servfd, &rin);
	    maxfd = servfd;
	}

	/* This loop check to see if a control socket has been marked for
	   closure.  If it is ready then it's closed */
	want_close = 0;
	for (last = NULL, ptr = options->Control_socks; ptr;) {
	    int close_ok, i;
	    close_ok = 0;
	    if (ptr->want_close) { /* Only check people who want it */
                LOG(LOG_TRACE, "Control socket %d wants close, lock=0x%x, active=%d\n", ptr->num, (int) ptr->lock, ptr->state.active);
		if (ptr->lock != NULL) { /* If the thread has had INIT */
		    int tmp;
		    /* Get the current number of active threads, if there
		       are no children left then free all resources
		       associated with it */
		    MutexLock(ptr->lock);
		    tmp = ptr->state.active; 
		    MutexUnlock(ptr->lock);
		    if (tmp <= 0) {
			close_ok = 1;
		    }
		} else {
		    /* if no INIT has been received we can close now */
		    close_ok = 1;
		}
		if (close_ok) { /* Close it */
                    LOG(LOG_TRACE, "Really closing control socket %d\n", ptr->num);
		    if (ptr->sync     != NULL) MutexDestroy(ptr->sync);
		    if (ptr->lock     != NULL) MutexDestroy(ptr->lock);
		    if (ptr->results  != NULL) SHMfree(SHM, ptr->results);

                    spec_zipf_free(&ptr->zipfdirseed);
                    spec_zipf_free(&ptr->zipffileseed);
                    spec_normal_free(&ptr->normalKAseed);
		    
		    while (ptr->url_deck.size > 0) {
			SHMfree(SHM, ptr->url_deck.deck[--ptr->url_deck.size]);
		    }
		    while (ptr->url_deck_spare.size > 0) {
			SHMfree(SHM, ptr->url_deck_spare.deck[--ptr->url_deck_spare.size]);
		    }
		    while (ptr->free_deck.size > 0) {
			SHMfree(SHM, ptr->free_deck.deck[--ptr->free_deck.size]);
		    }
		    if (ptr->load.url_root != NULL)
			SHMfree(SHM, ptr->load.url_root);
		    if (ptr->load.dynamic_url != NULL)
			SHMfree(SHM, ptr->load.dynamic_url);
		    if (ptr->load.dyn_get_script_url != NULL)
			SHMfree(SHM, ptr->load.dyn_get_script_url);
		    if (ptr->load.dyn_cookie_script_url != NULL)
			SHMfree(SHM, ptr->load.dyn_cookie_script_url);
		    if (ptr->load.dyn_cgi_script_url != NULL)
			SHMfree(SHM, ptr->load.dyn_cgi_script_url);
		    for (i = 0; i < ptr->load.num_classes; i++) {
			if (ptr->load.class[i] != NULL) {
			    if (ptr->load.class[i]->name != NULL)
				SHMfree(SHM, ptr->load.class[i]->name);
			    SHMfree(SHM, ptr->load.class[i]);
			}
		    }
		    SHMfree(SHM, ptr->url_deck.deck);
		    SHMfree(SHM, ptr->url_deck_spare.deck);
		    SHMfree(SHM, ptr->free_deck.deck);
		    for (lg_ptr = ptr->lg_data; lg_ptr != NULL;)
		    {
#ifdef WIN32
			    CloseHandle((HANDLE) lg_ptr->child);
#endif /* WIN32 */
			    lg_next = lg_ptr->next;
			    free(lg_ptr);
			    lg_ptr = lg_next;
		    }
#ifdef WIN32
                    (void) WSACloseEvent(ptr->mc_select_event); /*MC*/
                    (void) WSACloseEvent(ptr->mc_stop_event);   /*MC*/
#endif /* WIN32 */
		    LOG(LOG_TRACE, "Control Socket %d: close\n", ptr->num);
		    if (ptr->console_session) {
			close(ptr->fd);
		    } else {
			closesocket(ptr->fd);
		    }
		    if (last == NULL) {
			options->Control_socks = ptr->next;
			SHMfree(SHM, ptr);
			ptr = options->Control_socks;
		    } else {
			last->next = ptr->next;
			SHMfree(SHM, ptr);
			ptr = last->next;
		    }
SHMdump(SHM);
		} else {
		    /* Note that there is someone who wants to be closed
		       but isn't ready quite yet */
		    want_close = 1; 
		    last = ptr;
		    ptr = ptr->next;
		}
	    } else {
		last = ptr;
		ptr = ptr->next;
	    }
	}

	/* If we are in the process of dying, and there no control sockets
	   open, then we're done */
	if (dying == 1 && options->Control_socks == NULL) {
	    if (Host_lock != NULL)
		MutexDestroy(Host_lock);
	    SHMdestroy(SHM);
	    LOG(LOG_TRACE, "All my affairs are in order.  Farewell cruel world...\n");
#ifdef WIN32
            return 0; /* MC: let main() or MC_ServiceMain() handle exit */
#else
#ifdef DEBUG
    	    /* force core dump */
    	    abort();
#else
    	    exit(1);
#endif
#endif /* WIN32 */
	}

        gettime(&current_time); /* Get the current time for this loop */
	/* If we have no connections, and have waited too long for a new one
	   then abort */
	if (options->Control_socks == NULL && options->idle_timeout > 0 &&
		timeval_elapsed(&current_time, &lastaction_time) > 
		options->idle_timeout) {
	    LOG(LOG_TRACE, "We've been idle a while, terminating\n");
	    die_now = 1;
	}

	/* this loops through the control sockets adding them to the 
	   list of selectable sockets, and if autoadvance is on, it will
	   attempt to advance the time accordingly */
	for (ptr = options->Control_socks; ptr; ptr = ptr->next) {
	    if (ptr->auto_advance && ptr->next_mode != CS_NONE) {
		unsigned long elapsed_time, offset_time;

		/* Find out what our times are */
		elapsed_time = timeval_elapsed(&current_time, &ptr->start_time);
		offset_time  = timeval_elapsed(&ptr->start_time, epoch_ptr);

		/* While there is a next mode, and its time to use it */
		while (ptr->next_mode != CS_NONE && 
		       elapsed_time > ptr->next_event_time) {

		    /* Get the current state */
		    MutexLock(ptr->lock);

		    /* If we are going to result mode, wait until all of the
		       children have shut down */
		    if ( ptr->next_mode == CS_RESULT && ptr->state.active > 0) {
			/* Pause for 1/2 second */
			global_next_time = 500 + elapsed_time + offset_time;
			MutexUnlock(ptr->lock);
			break;
		    }

		    /* Advance the mode */
		    ptr->state.mode = ptr->next_mode;

		    /* Handle wait advance */
		    if (ptr->wait_advance && 
			    ptr->wait_advance <= ptr->state.mode) {
			/* Hey we reached our target!*/
			ptr->wait_advance = 0;
			ptr->input_mode=CS_INPUT_COMMAND;
			/* Need to parse a command now, we might have
			 * had one on the queue when we started waiting */
			ptr->need_command = 1;
			buf_printf(&ptr->output, "200 OK\n");
		    }

		    /* Write our new state */
		    MutexUnlock(ptr->lock);

		    /* Set what our next mode should be, and when we
		     * should be there */
		    ptr->next_event_time = 0;
		    switch(ptr->state.mode) {
		    case CS_RESULT:    ptr->next_mode = CS_NONE; break;
		    case CS_SHUTDOWN:  ptr->next_mode = CS_RESULT; break;
		    case CS_RAMPDOWN:  ptr->next_mode = CS_SHUTDOWN; break;
		    case CS_RUN:       ptr->next_mode = CS_RAMPDOWN; break;
		    case CS_RAMPUP:    ptr->next_mode = CS_RUN; break;
		    }
		    switch(ptr->next_mode) {
		    case CS_RESULT:   ptr->next_event_time+=500;
		    case CS_SHUTDOWN: ptr->next_event_time+=ptr->Rampdown_time;
		    case CS_RAMPDOWN: ptr->next_event_time+=ptr->Run_time;
		    case CS_RUN:      ptr->next_event_time+=ptr->Rampup_time;
		    }
		}

		/* Convert our local time into global time */
		if (ptr->next_event_time && 
		    ptr->next_event_time + offset_time < global_next_time) {
		    global_next_time = ptr->next_event_time + offset_time;
		}
	    }
	    if (!ptr->want_close) {
		/* If we aren't in the process of closing then put
		 * ourselves in the select masks */
		if (ptr->fd > maxfd)
		    maxfd = ptr->fd;
#ifdef WIN32
                mc_select_want = FD_CLOSE;

                if (ptr->input.len+ptr->input.start < sizeof(ptr->input.buffer))
                {
                    mc_select_want |= FD_READ;
		    FD_SET(ptr->fd, &rin);
                }
                if (ptr->output.len > 0)
                {
		    mc_select_want |= FD_WRITE;
		    FD_SET(ptr->fd, &win);
                }
                if (WSAEventSelect(ptr->fd, ptr->mc_select_event, mc_select_want) == SOCKET_ERROR)
                    die("WSAEventSelect() failed (control socket %d), errno = %d\n", ptr->num, WSAGetLastError());
                MC_Control_Events[MC_NControlEvents++] = ptr->mc_select_event;
#else
		/* Don't read for us unless there is space in the
		 * input buffer, I assume this makes it possible to
		 * lock up the client by sending a really long line */
		if (ptr->input.len+ptr->input.start < sizeof(ptr->input.buffer)) {
		    FD_SET(ptr->fd, &rin);

		}
		if (ptr->output.len > 0) {
		    FD_SET(ptr->fd, &win);
		}
#endif /* WIN32 */
	    }
	}

        gettime(&current_time); /* Get the current time for this loop */
	/* Figure out how long of a wait we have to pass to select */
        /* DEC needs it to be INT_MAX instead of LONG_MAX */
	if (global_next_time != INT_MAX || want_close || die_now ||
	    options->idle_timeout > 0 ) {
	    unsigned long wait_time, offset_time, tmp;

	    wait_time = 0;

	    offset_time  = timeval_elapsed(&current_time, epoch_ptr);
	    if (global_next_time > offset_time) {
		wait_time = global_next_time - offset_time;
	    } 

	    /* If we have an idle timeout, then make sure don't ignore it */
	    if (options->idle_timeout > 0 && options->Control_socks == NULL) {
		tmp = options->idle_timeout - 
			timeval_elapsed(&current_time, &lastaction_time);
		if (tmp < wait_time)
		    wait_time = tmp;
	    }

	    /* If we are trying to shut something down then poll at
	     * least every second */
	    if (wait_time > 1000 && want_close || die_now) {
		wait_time = 1000;
	    }
	    if (wait_time <= 0) {
		LOG (LOG_ERROR, "Short Wait\n");
		wait_time = 1000;
	    }

#ifdef WIN32
            /*
             * MC: wait_time is milliseconds, timer frequency is 100 nanosecond intervals.
             */
            mc_alarm_time.QuadPart = (__int64) wait_time * -10000;
            if (!SetWaitableTimer(MC_ControlSocketTimer, &mc_alarm_time, 0, NULL, NULL, FALSE))
                die("SetWaitableTimer() failed - %s\n", MC_ERRTEXT(mcbuf));
            MC_Control_Events[MC_NControlEvents++] = MC_ControlSocketTimer;
            mc_want_alarm_event = TRUE;
        }
        else
            mc_want_alarm_event = FALSE;
        if ((mcret = WaitForMultipleObjects(MC_NControlEvents, MC_Control_Events, FALSE, INFINITE)) == WAIT_FAILED)
            die("WaitForMultipleObjects() failed - %s\n", MC_ERRTEXT(mcbuf));
        if (mcret == WAIT_OBJECT_0)
            continue; /* the Stop event was signaled */
        else if (mc_want_alarm_event && mcret == (WAIT_OBJECT_0 + MC_NControlEvents - 1))
            continue; /* the Alarm event went off */
        /*
         * Our WaitForMultipleObjects() wasn't interrupted by the Stop event or 
         * the Alarm event so it must have been I/O ready on one of the sockets.
         */
        timeval_tmp.tv_sec = timeval_tmp.tv_usec = 0; /* poll */
        timeval_ptr = &timeval_tmp;
#else
	    timeval_tmp.tv_sec = wait_time / 1000;
	    timeval_tmp.tv_usec = (wait_time % 1000) * 1000;
	    timeval_ptr = &timeval_tmp;
	} else {
	    timeval_ptr = NULL;
	}
#endif /* WIN32 */

	LOG(LOG_SOCKET, "Selecting\n");
	/* These copies really are only necessary if we weren't going
	 * to generate the masks each time.  But they don't hurt much */
	memcpy(&rout, &rin, sizeof(rin));
	memcpy(&wout, &win, sizeof(win));
	memcpy(&eout, &ein, sizeof(ein));

	rc = select(maxfd+1, &rout, &wout, &eout, timeval_ptr);
	if (rc < 0) {
	    if (errno == EINTR || errno == EAGAIN || errno == 0)
		continue;
#if defined(_WIN32)
	    die ("Error in select: return=%d, error=%d\n", rc, WSAGetLastError());
#else
	    die ("Error in select: %s(%d)\n", strerror(errno), errno);
#endif
	} else if (rc != 0) { 
	    /* something happened, so update the idle timer */
	    gettime(&lastaction_time);
	}
	/* Check to see if we received a new control socket to control */
	if (FD_ISSET(servfd, &rout)) {
	    control_socket_t *cs;
	    SOCKET newsock;
	    LOG(LOG_SOCKET, "Accepting new control socket\n");
	    newsock_len = sizeof(newsock_addr);
	    newsock     = accept(servfd, (struct sockaddr FAR *)&newsock_addr, 
				    &newsock_len);
	    if (newsock < 0) {
		warn ("Error in accept: %s(%d)\n", strerror(errno), errno);
		continue;
	    }

	    /* Initialize the data */
	    cs = (control_socket_t *)SHMmalloc(SHM, sizeof(*cs));
	    if (cs == NULL) {
		die ("Error allocating %d bytes\n", sizeof(*cs));
	    }
	    memset(cs, 0, sizeof(*cs));
	    cs->state.mode = CS_SETUP;
	    cs->fd      = newsock;
	    cs->next    = options->Control_socks;
	    cs->num     = control_socket_count++;
	    cs->options = options;
	    cs->abortive_close = 0;

	    cs->Load_Generators = 2;
	    cs->Rampup_time = 3 * 1000;
	    cs->Run_time= 5 * 1000;
	    cs->load.load = 10;
	    cs->first_seed  = 2;
	    cs->HTTP_Protocol  = 11;

	    cs->url_deck.size  = 0;
	    cs->url_deck.max   = cs->options->deck_size;
	    cs->url_deck.deck  = SHMmalloc(SHM, cs->options->deck_size*sizeof(char *));
	    cs->url_deck_spare.size  = 0;
	    cs->url_deck_spare.max   = cs->options->deck_size;
	    cs->url_deck_spare.deck  = SHMmalloc(SHM, cs->options->deck_size*sizeof(char *));

	    cs->free_deck.size = 0;
	    cs->free_deck.max  = cs->options->deck_size*2;
	    cs->free_deck.deck = SHMmalloc(SHM, cs->options->deck_size*sizeof(char *)*2);

	    options->Control_socks = cs;
	    cs->next_mode = CS_NONE;
	    if (cs->next == NULL) {
		/* If we are creating the only connection, then reset
		   the epoch time so we won't overflow in 50+ days */
		getepoch(&options->epoch_time);
	    }
#ifdef WIN32
            cs->mc_select_event = CreateEvent(NULL, FALSE, FALSE, NULL);
            if (cs->mc_select_event == NULL)
                die("Unable to create control socket select event - %s\n", MC_ERRTEXT(mcbuf));
            cs->mc_stop_event = CreateEvent(NULL, TRUE, FALSE, NULL);
            if (cs->mc_stop_event == NULL)
                die("Unable to create control socket stop event - %s\n", MC_ERRTEXT(mcbuf));
#endif /* WIN32 */
	    buf_printf(&cs->output, "100 %s\n", ProgramName);
	}
	/* For each control socket, go do any work necessary */
	for (ptr = options->Control_socks; ptr; ptr = next) {
		next = ptr->next;
	    ControlAction(FD_ISSET(ptr->fd, &rout), FD_ISSET(ptr->fd, &wout),
	                  FD_ISSET(ptr->fd, &eout), ptr->need_command, ptr);
            if (FD_ISSET(ptr->fd, &rout) || FD_ISSET(ptr->fd, &wout) ||
                        FD_ISSET(ptr->fd, &eout))
                gettime(&lastaction_time); /* something happened */
	}
    }
    return 0;
}


/* begin */

int main(int argc, char *argv[]) {
#ifdef HAVE_RLIMIT
    struct rlimit rlp;
#endif
    int ret = 0;

    SUB(main, enter);

    progname = &argv[0];
#ifdef WIN32
    MC_Main_Options = &options;
#endif /* WIN32 */

#ifdef HAVE_RLIMIT
    if(getrlimit(RLIMIT_NOFILE, &rlp) != 0)
    {
        perror("Could not Get Current Limit for 'nofiles(descriptors)'");
    }
    else
    {
        rlp.rlim_cur = rlp.rlim_max;
        if(setrlimit(RLIMIT_NOFILE, &rlp) != 0)
        {
            perror("Could not set 'nofiles(descriptors)' to System Max");
        }
    }
#endif
    /* Get the epoch to initialize the time subroutines */
    getepoch(&options.epoch_time); 

#ifdef HTTPS_SUPPORT
    /* global ssl initialization */

    /* registers the available ciphers */
    SSL_library_init();	

    /* add entropy to the Pseudo Random Number Generator (PRNG) */
    RAND_seed(rnd_seed, sizeof rnd_seed);

    /* set up BIO file pointers for stderr and stdout */
    if (bio_err == NULL)
        bio_err=BIO_new_fp(stderr,BIO_NOCLOSE);
    if (bio_stdout == NULL)
        bio_stdout=BIO_new_fp(stdout,BIO_NOCLOSE);

    /* Constructor for the SSLv3 SSL_METHOD structure for a dedicated client */
    ssl_method=SSLv3_client_method();

    /* load ssl error messages */
    SSL_load_error_strings();

    /* create a new SSL_CTX object as framework to establish SSL enabled 
       connections */
    c_ctx=SSL_CTX_new(ssl_method);
    if (c_ctx == NULL) {
        ERR_print_errors(bio_err);
#ifdef DEBUG
        /* force core dump */
        abort();
#else
        /*goto end;*/ exit(1);
#endif
    }
/*#ifdef CIPHER_STRING*/
    /*if ( 0 == SSL_CTX_set_cipher_list(c_ctx, "RC4-SHA")){*/

    if ( 0 == SSL_CTX_set_cipher_list(c_ctx, CIPHER_STRING)){
	ERR_print_errors(bio_err);
#ifdef DEBUG
    /* force core dump */
        abort();
#else
	exit(1);
#endif
    }
	
/*#endif*/

#endif

    /* Set up the request handler chain */
    InitRequestSystem();
    InitRequestUser();

    if (ParseArguments(argc, argv, &options))
            exit(0);
#ifdef WIN32 /*MC*/
    if (options.mc_svc_action != MC_SVC_NONE)
    {
        if ((options.mc_svc_action & MC_SVC_INSTALL) && !MC_CmdInstallService(argv[0]))
            return 1;
        if ((options.mc_svc_action & MC_SVC_START)   && !MC_CmdStartService(argc, argv))
            return 1;
        if ((options.mc_svc_action & MC_SVC_STOP)    && !MC_CmdStopService())
            return 1;
        if ((options.mc_svc_action & MC_SVC_REMOVE)  && !MC_CmdRemoveService())
            return 1;
        return 0; /* success */
    }
#endif /* WIN32 */
    if (InitChildLayer())
	die("Error initializing child layer\n");
    if (InitializeThreadSystem(&options))
	die("Error initializing thread system\n");
#ifdef WIN32 /* MC*/
    if (options.daemonize)
    {
	/* StartServiceCtrlDispatcher() returns on service stop */
        if (!StartServiceCtrlDispatcher(mc_svc_dispatch_table))
	{
		MC_AddToMessageLog(EVENTLOG_ERROR_TYPE, "StartServiceCtrlDispatcher failed.");
		ret = 1;
	}
    }
    else
    {
    if (!MC_Win32Initialize(&options))
        return 1;
    SetConsoleCtrlHandler(MC_ConsoleControlHandler, TRUE);	/* handle CTRL+C or CTRL+BREAK */
    if (InitializeWorkDir(&options))
	die("Error initializing working directory\n");
#else
    if (InitializeSignals(&options))
	die("Error initializing signal handlers\n");
    if (InitializeWorkDir(&options))
	die("Error initializing working directory\n");
    if (options.daemonize)
	InitializeDaemon(&options);
#endif /* WIN32 */
    if (InitializeControlPort(&options))
	die("Error initializing control port\n");

#if !defined(_WIN32)    
    if (options.console_session)
	Setup_STDIN_cs(&options);
#endif

    if (HandleControlActivity(&options))
	die("Error handling control activity\n");
#ifdef WIN32
    }
#endif
    free(options.workdir);
    closesocket(options.Control_fd);
    return ret;
}

/* Set up stdin as a control socket.  Useful for debugging */
int Setup_STDIN_cs(cl_options_t *options) {
    control_socket_t *cs;

    SUB(Setup_STDIN_cs, enter);

    cs = (control_socket_t *)SHMmalloc(SHM, sizeof(*cs));
    if (cs == NULL) 
	die ("Error allocating %d bytes\n");

    memset(cs, 0, sizeof(*cs));

    cs->console_session = 1;
    cs->fd         = 0;
    cs->num        = control_socket_count++;
    cs->next       = options->Control_socks;
    cs->options    = options;
    cs->next_mode  = CS_NONE;
    cs->state.mode = CS_SETUP;
    cs->abortive_close = 0;
    options->Control_socks = cs;
    cs->first_seed  = 2;
    cs->HTTP_Protocol  = 11;
    cs->load.load = 10;
#if 0

    cs->Run_time = 10000;
    cs->Rampup_time = 5000;
    cs->Rampdown_time = 3000;
    cs->Load_Generators = 1;
    cs->auto_advance = 1;
    strcpy(cs->Client_name, "foo");
    strcpy(cs->Server_name, "bar");
#endif
    buf_printf(&cs->output, "100 %s\n", ProgramName);
    return 0;
}
