/* (C) 1997 Standard Performance Evaluation Corporation */
#include "client.h"

static char *parse_c_rcsid = "$Id: parse.c,v 1.7 1998/02/25 01:46:01 channui Exp $";

extern void load_generator(void *ptr);
extern void *SHM;

#ifdef WIN32
extern int MC_parse_ping(word_list *words, control_socket_t *cs);
extern int MC_parse_fetchurl(word_list *words, control_socket_t *cs);
#endif /* WIN32 */

/* The parse code receives a request in the form of an array of
 * character strings with one whitespace seperated word per entry.
 * When the program is initialized it registers a series of "handlers".
 * New handlers are prepended to the beginning of the list.
 * Once a request is received, each handler is called in turn.  If the
 * handler cannot process the request it returns with PARSE_NO_MATCH.
 * Usage information can be requested by passing a null value to a
 * handler.  Once a match or error is found, no other entries will be
 * processed. */

/* Just a linked list of handlers */
typedef struct request_handler_s {
    struct request_handler_s *next;
    int (*func)(word_list *words, control_socket_t *cs);
} request_handler_t;

request_handler_t *request_list = NULL;

/* A ASCII table of names to map to the state ids */
char *cs_state_labels[] = { "ERROR", "SETUP", "SYNC", "RAMPUP", "RUN", "RAMPDOWN", "SHUTDOWN", "RESULT", "NONE" };

/* check_parse_args is just a convenience function to process common
 * cases, specifically not matching, wrong # of args, and usage
 * requests */
int check_parse_args (word_list *words, control_socket_t *cs, char *command, 
                      int num_args, char *usage) {
    SUB(check_parse_args , enter);

    if (words == NULL) { /* Usage Request */
	buf_printf(&cs->output, "%s\r\n", usage);
	return PARSE_USAGE;
    }

    if (strcasecmp(words->word[0], command)) /* Not me */
	return PARSE_NO_MATCH;

    if (num_args != 0 && words->count != num_args) { /* Wrong # of args */
	buf_printf(&cs->output, "400 USAGE: %s\r\n", usage);
	return PARSE_ERROR;
    }

    return PARSE_OK;
}

/* This routine breaks up a string into individual "words" the string
   ends at a \r\n or \0 character.  The caller is responsible for
   freeing the data */
int break_on_whitespace(char *text, word_list **buffer) {
    char **ind;
    int len, count, num_words;
    word_list *words;
    char *ptr;

    SUB(break_on_whitespace, enter);

    num_words = 0;

    /* This function is called from the input routine, so we only parse
     * up to a newline.  This is bad, but will work for now */
    len = strcspn(text, "\n");
    if (len == 0) 
	return 0;

    /* Can't have more than 1/2 of the characters be interesting "words" */
    *buffer = (word_list *)malloc(sizeof(char *) * len / 2 + sizeof(*words));
    if (*buffer == NULL)
	die ("Error allocating %d bytes\r\n");
    words = *buffer;
    ind   = words->word;

    ptr = text;
    while (*ptr != '\n' && *ptr != '\0') {
	count = strcspn(ptr, " \t\r\n");
	if (count) {
	    *ind = (char *)malloc(count+1);
	    if (*ind == NULL)
		die ("Error allocating %d bytes\r\n");
	    /* Not quite what I want, but it will do for now */
	    if (*ptr == '"' && *(ptr+count-1) == '"') {
		memcpy(*ind, ptr+1, count-2);
		*((*ind)+count-2) = 0;
	    } else {
		memcpy(*ind, ptr, count);
		*((*ind)+count) = 0;
	    }
	    num_words++;
	    ind++;
	    ptr += count;

	}
	count = strspn(ptr, " \t\r");
	ptr += count;
    }
    *ind = NULL;
    if (!num_words) {
	free(*buffer);
    }
    words->count = num_words;
    return num_words;
}

/* Easy way to get rid of those annoying buffers break_on_whitespace
 * creates */
int free_whitespace_buf(word_list *list) {
    int i;

    SUB(free_whitespace_buf, enter);
    if (list == NULL)
	return 0;
    for (i = list->count - 1; i >= 0; i--) {
	free(list->word[i]);
    }
    free (list);
    return 0;
}

/* This is unimplemented, it should close all of the file descriptors
 * and then restart everything, probably with an exec in order to get
 * rid of memory we've allocated *TODO* */
int parse_restart (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_restart, enter);
    rc = check_parse_args(words, cs, "RESTART", 1, "RESTART");
    if (rc != PARSE_OK) return rc;

    /* TODO: Schedule a restart here */
    buf_printf(&cs->output, "RESTART not implemented yet\r\n");
    return PARSE_ERROR;
}


/* Reset back to setup state after a run.  This function needs to clean
 * up a bit more, releasing Shared resources and ensuring that any
 * children are killed off *TODO* */
int parse_reset (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_reset, enter);
    rc = check_parse_args(words, cs, "RESET", 1, "RESET");
    if (rc != PARSE_OK) return rc;

    /* TODO: Need to kill off children, reset sockets, etc */
    MutexLock(cs->lock);
    cs->state.mode = CS_SETUP;
    MutexUnlock(cs->lock);
    return PARSE_OK;
}

/* close this connection */
int parse_quit (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_quit, enter);
    rc = check_parse_args(words, cs, "QUIT", 1, "QUIT");
    if (rc != PARSE_OK) return rc;

    cs->must_exit = 1;
    return PARSE_OK;
}

/* Kill off this daemon */
int parse_terminate (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_quit, enter);
    rc = check_parse_args(words, cs, "TERMINATE", 1, "TERMINATE");
    if (rc != PARSE_OK) return rc;

    LOG(LOG_TRACE, "Terminate command received.  Will try to shut down\n");
#ifdef WIN32
    MC_ShutdownClient(); /*MC*/
#else
    die_now=2;
#endif /* WIN32 */
    cs->must_exit = 1;
    return PARSE_OK;
}

/* Print out information on our state */
/* Need to expand to included other more descriptive fields, like load *TODO* */
int parse_query (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_query, enter);
    rc = check_parse_args(words, cs, "QUERY", 1, "QUERY");
    if (rc != PARSE_OK) return rc;

    buf_printf(&cs->output, "Client Name:     %s\r\n",  cs->Client_name);
    buf_printf(&cs->output, "Server Name:     %s\r\n",  cs->Server_name);
    buf_printf(&cs->output, "Rampup Time:     %lu\r\n", cs->Rampup_time/1000);
    buf_printf(&cs->output, "Run Time:        %lu\r\n", cs->Run_time/1000);
    buf_printf(&cs->output, "Rampdown Time:   %lu\r\n", cs->Rampdown_time/1000);
    buf_printf(&cs->output, "Load Generators: %lu\r\n", cs->Load_Generators);
    buf_printf(&cs->output, "Load:            %f\r\n",  cs->load.load);
    buf_printf(&cs->output, "State:           %s\r\n",  cs_state_labels[cs->state.mode]);
    buf_printf(&cs->output, "Auto Advance:    %d\r\n",  cs->auto_advance);
    buf_printf(&cs->output, "Active LGs:      %lu\r\n", cs->state.active);
    return PARSE_OK;
}

/* Start up the children and basically start things rolling */
int parse_init (word_list *words, control_socket_t *cs) {
    int error = 1;
    int i;
    int rc;

    SUB(parse_init, enter);
    rc = check_parse_args(words, cs, "INIT", 1, "INIT");
    if (rc != PARSE_OK) return rc;


    if (cs->state.mode != CS_SETUP) {
	buf_printf(&cs->output, "500 Run in progress\r\n");
    } else if (cs->Load_Generators <= 0) {
	buf_printf(&cs->output, "501 Number of Load Generators not specified\r\n");
#if 0
    } else if (cs->Client_name[0] == '\0') {
	buf_printf(&cs->output, "501 Client Name not specified\r\n");
    } else if (cs->Server_name[0] == '\0') {
	buf_printf(&cs->output, "501 Server Name not specified\r\n");
#endif
    } else if (cs->auto_advance && cs->Run_time <= 0) {
	buf_printf(&cs->output, "501 Run Time not specified\r\n");
    } else if (cs->Load_Generators <= 0) {
	buf_printf(&cs->output, "501 Load Generators not specified\r\n");
    } else {
	error = 0;
    }
    if (error) return PARSE_ERROR;

    /* Create all of our synchronization and shared resources */
    if (cs->lock != NULL) { MutexDestroy(cs->lock); cs->lock = NULL; }
    cs->lock = MutexCreate();
    if (cs->lock == NULL) die ("parse_init: MutexCreate failed!\n");
    if (cs->sync != NULL) { MutexDestroy(cs->sync); cs->sync = NULL; }
    cs->sync = MutexCreate();
    if (cs->sync == NULL) die ("parse_init: MutexCreate failed!\n");
    MutexLock(cs->sync);
    if (cs->results) SHMfree(SHM, cs->results);
    cs->results = SHMmalloc(SHM, sizeof(struct UserResults)*cs->Load_Generators);
    if (cs->results == NULL) die ("parse_init: SHMmalloc failed!\n");

    /* Initialize the random seeds */
    spec_srandom(&cs->ranseed, cs->first_seed);
    spec_zipf_setup(&cs->zipfdirseed,  &cs->ranseed, cs->load.max_dirs, 1.0);
    spec_zipf_setup(&cs->zipffileseed, &cs->ranseed, FILES_PER_CLASS,   1.0);
    spec_normal_setup(&cs->normalKAseed, &cs->ranseed,cs->load.keep_alive,3.5); 

    /* Update our state */
    cs->state.mode = CS_SYNC;

#ifdef HTTPS_SUPPORT
#if defined( USE_POSIX_THREADS ) || defined ( WIN32 )
    https_thread_setup();
#endif
#endif

    /* Spawn off the children */
    LOG(LOG_THREAD, "parse:  Load_Generators = %d \n", cs->Load_Generators);
    for (i = 0; i < cs->Load_Generators; i++) {
	lg_data_t *ptr;

	/* Make load generate data to send to client */
	ptr = (lg_data_t *)malloc(sizeof(*ptr));
	if (ptr == NULL)
	    die ("parse_init: Can't malloc %d bytes!\n", sizeof(*ptr));
	memset(ptr, 0, sizeof(*ptr));
	ptr->cs = cs;
	MutexLock(cs->lock);
	ptr->next = ptr->cs->lg_data;
	ptr->cs->lg_data = ptr;
	MutexUnlock(cs->lock);
	ptr->num = i;

	ptr->child = CreateChild(load_generator, ptr);
    }

    /* Really we don't need to lock here, but it's good practice anyway */
    /* Wait for all clients to start up */
    do {
	MutexLock(cs->lock);
	LOG(LOG_THREAD, "Active Clients: %d\n", cs->state.active);
	MutexUnlock(cs->lock);
	spec_msleep(100);
    }  while (cs->state.active < cs->Load_Generators);

    /* If we are an auto advance connection then start advancing */
    if (cs->auto_advance) {
	/* Update the state */
	MutexLock(cs->lock);
	cs->state.mode = (cs->Rampup_time>0)?CS_RAMPUP:CS_RUN;
	MutexUnlock(cs->lock);

	/* Release our children */
	gettime(&cs->start_time);
	MutexUnlock(cs->sync);

	/* Get ready for next advance */
	cs->next_event_time = cs->Rampup_time;
	cs->next_mode       = CS_RUN;
	if (cs->state.mode == CS_RUN) {
	    cs->next_event_time += cs->Run_time;
	    cs->next_mode        = CS_RAMPDOWN;
	}
    }

    return PARSE_OK;
}

/* Start up the clients from sync state. If auto advancing then you
 * *must* call this or init.  */
int parse_dorampup (word_list *words, control_socket_t *cs) {
    int need_unlock_sync = 0;
    int rc;

    SUB(parse_dorampup, enter);
    rc = check_parse_args(words, cs, "DORAMPUP", 1, "DORAMPUP");
    if (rc != PARSE_OK) return rc;

    if (cs->state.mode != CS_SYNC) {
	buf_printf(&cs->output, "500 Not in SYNC mode\r\n");
	return 1;
    }

    /* Advance our state */
    MutexLock(cs->lock);
    need_unlock_sync = (cs->state.mode == CS_SYNC);
    cs->state.mode = (cs->Rampup_time>0)?CS_RAMPUP:CS_RUN;
    MutexUnlock(cs->lock);
    gettime(&cs->start_time);

    /* Release clients */
    if (need_unlock_sync)
	MutexUnlock(cs->sync);

    /* Set information for next advance */
    if (cs->auto_advance) {
	cs->next_event_time = cs->Rampup_time;
	cs->next_mode       = CS_RUN;
	if (cs->state.mode == CS_RUN) {
	    cs->next_event_time += cs->Run_time;
	    cs->next_mode        = CS_RAMPDOWN;
	}
    }
    return PARSE_OK;
}

/* Move to run state */
int parse_dorun (word_list *words, control_socket_t *cs) {
    int need_unlock_sync = 0;
    int rc;

    SUB(parse_dorun, enter);

    rc = check_parse_args(words, cs, "DORUN", 1, "DORUN");
    if (rc != PARSE_OK) return rc;

    if (cs->state.mode != CS_SYNC && cs->state.mode != CS_RAMPUP) {
	buf_printf(&cs->output, "500 Not in SYNC mode\r\n");
	return 1;
    }
    need_unlock_sync = (cs->state.mode == CS_SYNC);

    MutexLock(cs->lock);
    cs->state.mode = CS_RUN;
    MutexUnlock(cs->lock);

    if (need_unlock_sync)
	MutexUnlock(cs->sync);

    return PARSE_OK;
}

/* Move to rampdown state */
int parse_dorampdown (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_dorampdown, enter);

    rc = check_parse_args(words, cs, "DORAMPDOWN", 1, "DORAMPDOWN");
    if (rc != PARSE_OK) return rc;

    if (cs->state.mode != CS_RUN) {
	buf_printf(&cs->output, "500 Not in RUN mode\r\n");
	return 1;
    }

    MutexLock(cs->lock);
    cs->state.mode = CS_RAMPDOWN;
    MutexUnlock(cs->lock);

    return PARSE_OK;
}

/* Move to shutdown state */
int parse_doshutdown (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_doshutdown, enter);
    rc = check_parse_args(words, cs, "DOSHUTDOWN", 1, "DOSHUTDOWN");
    if (rc != PARSE_OK) return rc;

    if (cs->state.mode != CS_RUN && cs->state.mode != CS_RAMPDOWN ) {
	buf_printf(&cs->output, "500 Not in RUN mode\r\n");
	return 1;
    }

    MutexLock(cs->lock);
    cs->state.mode = CS_SHUTDOWN;
    MutexUnlock(cs->lock);

    return PARSE_OK;
}


int parse_result (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_result, enter);
    rc = check_parse_args(words, cs, "RESULT", 1, "RESULT");
    if (rc != PARSE_OK) return rc;

    buf_printf(&cs->output, "foobarbaz\n");
    return PARSE_OK;
}

/* Run through and print all of the usage data */
int parse_help (word_list *words, control_socket_t *cs) {
    request_handler_t *ptr;
    int rc;

    SUB(parse_help, enter);

    rc = check_parse_args(words, cs, "HELP", 1, "HELP");
    if (rc != PARSE_OK) return rc;

    for (ptr = request_list; ptr; ptr=ptr->next) {
	ptr->func(NULL, cs);
    }
    return PARSE_OK;
}

/* Set the client name.  This isn't actually used anywhere so could be
 * removed */
int parse_client(word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_client, enter);
    rc = check_parse_args(words, cs, "CLIENT", 2, "CLIENT [name]");
    if (rc != PARSE_OK) return rc;

    if (strlen(words->word[1]) >= sizeof(cs->Client_name)) {
	buf_printf(&cs->output, 
	    "401 ERROR: STRING TOO LONG: %d character maximum\r\n", sizeof(cs->Client_name));
	return PARSE_ERROR;
    }

    strncpy(cs->Client_name, words->word[1], sizeof(cs->Client_name));
    return PARSE_OK;
}

int parse_clientnum(word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_client, enter);
    rc = check_parse_args(words, cs, "CLIENTNUM", 2, "CLIENTNUM [number]");
    if (rc != PARSE_OK) return rc;

    cs->Client_num = atoi(words->word[1]);
    cs->my_last_ad = -1;
    return PARSE_OK;
}

int parse_http_protocol(word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_client, enter);
    rc = check_parse_args(words, cs, "HTTPPROTO", 2, "HTTPPROTO [level*10]");
    if (rc != PARSE_OK) return rc;

    cs->HTTP_Protocol = atoi(words->word[1]);
    return PARSE_OK;
}

/* Set the server name.  This isn't actually used anywhere so could be
 * removed */
int parse_server(word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_server, enter);
    rc = check_parse_args(words, cs, "SERVER", 2, "SERVER [name]");
    if (rc != PARSE_OK) return rc;

    if (strlen(words->word[1]) >= sizeof(cs->Server_name)) {
	buf_printf(&cs->output, 
	    "401 ERROR: STRING TOO LONG: %d character maximum\r\n", sizeof(cs->Server_name));
	return PARSE_ERROR;
    }

    strncpy(cs->Server_name, words->word[1], sizeof(cs->Server_name));
    return PARSE_OK;
}

/* Get the rampup time value */
int parse_rampup(word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_rampup, enter);
    rc = check_parse_args(words, cs, "RAMPUP", 2, "RAMPUP [time]");
    if (rc != PARSE_OK) return rc;

    cs->Rampup_time = atoi(words->word[1])*1000;
    return PARSE_OK;
}

/* Get the rampdown time value */
int parse_rampdown(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_rampdown, enter);
    rc = check_parse_args(words, cs, "RAMPDOWN", 2, "RAMPDOWN [time]");
    if (rc != PARSE_OK) return rc;

    cs->Rampdown_time = atoi(words->word[1])*1000;
    return PARSE_OK;
}

/* Get the run time value */
int parse_run(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_run, enter);
    rc = check_parse_args(words, cs, "RUN", 2, "RUN [time]");
    if (rc != PARSE_OK) return rc;

    cs->Run_time = atoi(words->word[1])*1000;
    return PARSE_OK;
}

/* Get the number of load generators value */
int parse_loadgen(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_loadgen, enter);
    rc = check_parse_args(words, cs, "LOADGEN", 2, "LOADGEN [num]");
    if (rc != PARSE_OK) return rc;

    cs->Load_Generators = atoi(words->word[1]);
    return PARSE_OK;
}

/* Get the autoadvance value (0 is off, any other number is on) */
int parse_autoadv(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_autoadv, enter);
    rc = check_parse_args(words, cs, "AUTOADV", 2, "AUTOADV [num]");
    if (rc != PARSE_OK) return rc;

    cs->auto_advance = atoi(words->word[1]);
    return PARSE_OK;
}

/* Get the autoadvance value (0 is off, any other number is on) */
int parse_abortive_close(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_autoadv, enter);
    rc = check_parse_args(words, cs, "ABORTIVE_CLOSE", 2, "ABORTIVE_CLOSE [num]");
    if (rc != PARSE_OK) return rc;

    cs->abortive_close = atoi(words->word[1]);
    return PARSE_OK;
}

/* Write a file */
int parse_writefile(word_list *words, control_socket_t *cs) {
    int rc;
    SUB(parse_writefile, enter);
    rc = check_parse_args(words, cs, "WRITEFILE", 2, "WRITEFILE [file]");
    if (rc != PARSE_OK) return rc;

    /* Don't allow users to use any odd characters.  This is so people
     * cannot write to random files outside of the work directory */
    if (strspn(words->word[1],
	"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVXYZ0123456789-_.")
	!= strlen(words->word[1])) {
	buf_printf(&cs->output, "404 ERROR opening file '%s': Filename must be [a-zA-Z0-9_-]+\r\n", words->word[1], strerror(errno));
	return PARSE_ERROR;
    }

    /* Open the file */
    cs->input_file = open(words->word[1], O_CREAT|O_TRUNC|O_RDWR, 0644);
    if (cs->input_file < 0) {
	buf_printf(&cs->output, "401 ERROR opening file '%s': %s\r\n", words->word[1], strerror(errno));
	return PARSE_ERROR;
    }

    /* This makes the input routine "do the right thing" */
    cs->input_mode = CS_INPUT_FILE;
    return PARSE_DEFER;
}

/* Reads a file from the work directory */
int parse_readfile(word_list *words, control_socket_t *cs) {
    int rc, last;
    int fd;
    char buf[4096];

    SUB(parse_readfile, enter);
    rc = check_parse_args(words, cs, "READFILE", 2, "READFILE [file]");
    if (rc != PARSE_OK) return rc;

    /* No odd characters allowed */
    if (strspn(words->word[1],
	"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVXYZ0123456789-_.")
	!= strlen(words->word[1])) {
	buf_printf(&cs->output, "404 ERROR opening file '%s': Filename must be [a-zA-Z0-9_-]+\r\n", words->word[1], strerror(errno));
	return PARSE_ERROR;
    }

    /* Open the file to read */
    fd = open(words->word[1], O_RDONLY);
    if (fd < 0) {
	buf_printf(&cs->output, "401 ERROR opening file '%s': %s\r\n", words->word[1], strerror(errno));
	return PARSE_ERROR;
    }

    /* Put it on the output buffer, it is important the file is small
     * so the output buffer doesn't overflow, this should probably
     * schedule the file to be written instead of dumping it all at
     * once. */
    while ((rc = read(fd, buf, sizeof(buf)))) {
	buf_write(&cs->output, buf, rc);
	last = rc;
    }

    /* Always make sure that the last line has a newline on it */
    if (buf[last-1] != '\n') {
	buf_write(&cs->output, "\n", 1);
    }

    /* The line consisting of a '.' signifies then end of a file.  If
     * you have one of these in your file then you need to fix it
     * before requesting it */
    buf_write(&cs->output, ".\n", 2);
    return PARSE_OK;
}

/* Waitadv sets things up to pause until the indicated state is reached */
int parse_waitadv(word_list *words, control_socket_t *cs) {
    int rc;
    int i, tmp;

    SUB(parse_waitadv, enter);
    rc = check_parse_args(words, cs, "WAITADV", 2, "WAITADV [mode]");
    if (rc != PARSE_OK) return rc;

    /* Find the requested state */
    tmp = 0;
    for (i = 0; i < sizeof(cs_state_labels)/sizeof(char *); i++) {
	if (strcasecmp(words->word[1], cs_state_labels[i]) == 0) {
	    tmp = i;
	}
    }
    if (tmp == 0) {
	buf_printf(&cs->output, "404 Invalid option: %s\r\n", words->word[1]);
	return PARSE_ERROR;
    }

    /* Make sure it's valid to do this */
    if (cs->auto_advance == 0) {
	buf_printf(&cs->output, "405 Auto Advance is not enabled\r\n");
	return PARSE_ERROR;
    }
    if (cs->state.mode < CS_RAMPUP && cs->state.mode != CS_NONE) {
	buf_printf(&cs->output, "406 Not running\r\n");
	return PARSE_ERROR;
    }

    /* We're already there */
    if (tmp <= cs->state.mode || tmp == CS_NONE) {
	cs->wait_advance = 0;
	return PARSE_OK;
    }

    /* Ok, start waiting */
    cs->wait_advance = tmp;
    cs->input_mode = CS_INPUT_NONE;
    return PARSE_DEFER;
}

/* Get the base random seed for this control socket */
int parse_srandom (word_list *words, control_socket_t *cs) {
    int rc;

    SUB(parse_dorampdown, enter);

    rc = check_parse_args(words, cs, "SRANDOM", 2, "SRANDOM [int]");
    if (rc != PARSE_OK) return rc;
    
    cs->first_seed = atoi(words->word[1]);
    spec_srandom(&ranseed, cs->first_seed);
    return PARSE_OK;
}

/* Just prepends the handler onto the list of handlers */
void add_request_handler(int (*func)(word_list *words, control_socket_t *cs)) {
    request_handler_t *node;

    SUB(add_request_handler, enter);
    node = (request_handler_t *)malloc(sizeof(request_handler_t));
    if (node == NULL)
	die ("add_request_handler: failed to malloc %d bytes\r\n", 
	     sizeof(request_handler_t));
    node->next = request_list;
    node->func = func;
    request_list = node;
}

/* This gets called at the beginning of the program.  It registers all
 * of the non-load specific handlers */
int InitRequestSystem() {
    SUB(InitRequestSystem, enter);
    add_request_handler(&parse_autoadv);
    add_request_handler(&parse_client);
    add_request_handler(&parse_clientnum);
    add_request_handler(&parse_dorampdown);
    add_request_handler(&parse_dorampup);
    add_request_handler(&parse_dorun);
    add_request_handler(&parse_doshutdown);
    add_request_handler(&parse_help);
    add_request_handler(&parse_init);
    add_request_handler(&parse_loadgen);
    add_request_handler(&parse_query);
    add_request_handler(&parse_quit);
    add_request_handler(&parse_rampdown);
    add_request_handler(&parse_rampup);
    add_request_handler(&parse_reset);
    add_request_handler(&parse_restart);
    add_request_handler(&parse_result);
    add_request_handler(&parse_run);
    add_request_handler(&parse_server);
    add_request_handler(&parse_waitadv);
    add_request_handler(&parse_writefile);
    add_request_handler(&parse_readfile);
    add_request_handler(&parse_srandom);
    add_request_handler(&parse_terminate);
    add_request_handler(&parse_abortive_close);
    add_request_handler(&parse_http_protocol);
#ifdef WIN32
    add_request_handler(&MC_parse_ping);
    add_request_handler(&MC_parse_fetchurl);
#endif /*WIN32*/
    return 0;
}

/* Run through all of the handlers and try to find one that fits. */
int ParseRequest(word_list *words, control_socket_t *cs) {
    request_handler_t *ptr;
    int rc = PARSE_NO_MATCH;

    SUB(ParseRequest, enter);
    for (ptr = request_list; ptr; ptr=ptr->next) {
	rc = ptr->func(words, cs);
	if (rc == PARSE_OK) {
	    LOG(LOG_OUT, "200 OK\r\n");
	}
	if (rc) return rc;
    }

    LOG(LOG_ERROR|LOG_OUT, "401 ERROR: UNKNOWN COMMAND\r\n");
    return 1;
}
